/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.orient.internal.freeze;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.eventbus.Subscribe;
import com.orientechnologies.common.concur.lock.OModificationOperationProhibitedException;
import com.orientechnologies.orient.server.OServer;
import com.orientechnologies.orient.server.distributed.ODistributedConfiguration;
import com.orientechnologies.orient.server.distributed.ODistributedServerManager;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Provider;
import javax.inject.Singleton;
import org.apache.shiro.authz.Permission;
import org.joda.time.DateTimeZone;
import org.sonatype.nexus.common.app.Freezable;
import org.sonatype.nexus.common.app.ManagedLifecycle;
import org.sonatype.nexus.common.event.EventAware;
import org.sonatype.nexus.common.event.EventManager;
import org.sonatype.nexus.common.node.NodeAccess;
import org.sonatype.nexus.common.node.NodeMergedEvent;
import org.sonatype.nexus.common.stateguard.Guarded;
import org.sonatype.nexus.common.stateguard.StateGuardLifecycleSupport;
import org.sonatype.nexus.orient.DatabaseInstance;
import org.sonatype.nexus.orient.freeze.DatabaseFreezeChangeEvent;
import org.sonatype.nexus.orient.freeze.DatabaseFreezeService;
import org.sonatype.nexus.orient.freeze.DatabaseFrozenStateManager;
import org.sonatype.nexus.orient.freeze.FreezeRequest;
import org.sonatype.nexus.orient.freeze.ReadOnlyState;
import org.sonatype.nexus.orient.internal.freeze.DefaultReadOnlyState;
import org.sonatype.nexus.security.SecurityHelper;
import org.sonatype.nexus.security.privilege.ApplicationPermission;

@Named
@ManagedLifecycle(phase=ManagedLifecycle.Phase.STORAGE)
@Singleton
public class DatabaseFreezeServiceImpl
extends StateGuardLifecycleSupport
implements DatabaseFreezeService,
EventAware,
EventAware.Asynchronous {
    public static final String SERVER_NAME = "*";
    static final ApplicationPermission READ_ONLY_PERMISSION = new ApplicationPermission("*", Arrays.asList("read"));
    private final Set<Provider<DatabaseInstance>> providers;
    private final EventManager eventManager;
    private final DatabaseFrozenStateManager databaseFrozenStateManager;
    private final Provider<ODistributedServerManager> distributedServerManagerProvider;
    private final NodeAccess nodeAccess;
    private final SecurityHelper securityHelper;
    private final List<Freezable> freezables;

    @Inject
    public DatabaseFreezeServiceImpl(Set<Provider<DatabaseInstance>> providers, EventManager eventManager, DatabaseFrozenStateManager databaseFrozenStateManager, Provider<OServer> server, NodeAccess nodeAccess, SecurityHelper securityHelper, List<Freezable> freezables) {
        this.providers = (Set)Preconditions.checkNotNull(providers);
        this.eventManager = (EventManager)Preconditions.checkNotNull((Object)eventManager);
        this.databaseFrozenStateManager = (DatabaseFrozenStateManager)Preconditions.checkNotNull((Object)databaseFrozenStateManager);
        Preconditions.checkNotNull(server);
        this.distributedServerManagerProvider = () -> ((OServer)server.get()).getDistributedManager();
        this.nodeAccess = (NodeAccess)Preconditions.checkNotNull((Object)nodeAccess);
        this.securityHelper = securityHelper;
        this.freezables = (List)Preconditions.checkNotNull(freezables);
    }

    protected void doStart() {
        List<FreezeRequest> state = this.databaseFrozenStateManager.getState();
        if (!state.isEmpty()) {
            this.refreezeOnStartup(state);
        }
    }

    @Override
    @Guarded(by={"STARTED"})
    public boolean isFrozen() {
        if (this.nodeAccess.isClustered()) {
            ODistributedServerManager serverManager = (ODistributedServerManager)this.distributedServerManagerProvider.get();
            HashMap<String, ODistributedConfiguration.ROLES> roles = new HashMap<String, ODistributedConfiguration.ROLES>();
            for (String database : serverManager.getMessageService().getDatabases()) {
                ODistributedConfiguration configuration = serverManager.getDatabaseConfiguration(database);
                roles.put(database, configuration.getServerRole(SERVER_NAME));
            }
            return roles.values().stream().allMatch(r -> ODistributedConfiguration.ROLES.REPLICA.equals(r));
        }
        return this.providers.stream().allMatch(provider -> ((DatabaseInstance)provider.get()).isFrozen());
    }

    @Override
    @Guarded(by={"STARTED"})
    public void checkUnfrozen() {
        this.checkUnfrozen("Database is frozen, unable to proceed.");
    }

    @Override
    @Guarded(by={"STARTED"})
    public void checkUnfrozen(String message) {
        if (this.isFrozen()) {
            throw new OModificationOperationProhibitedException(message);
        }
    }

    @Override
    @Guarded(by={"STARTED"})
    public synchronized FreezeRequest requestFreeze(FreezeRequest.InitiatorType type, String initiatorId) {
        FreezeRequest result;
        if (FreezeRequest.InitiatorType.USER_INITIATED.equals((Object)type) && !this.getState().isEmpty()) {
            this.log.warn("rejecting {} request for {} as 1 or more requests are already present in {}", new Object[]{type, initiatorId, this.getState()});
            return null;
        }
        boolean frozenStatePrior = this.isFrozen();
        FreezeRequest request = new FreezeRequest(type, initiatorId);
        if (this.nodeAccess.isClustered()) {
            request.setNodeId(this.nodeAccess.getId());
        }
        if ((result = this.databaseFrozenStateManager.add(request)) != null && !frozenStatePrior) {
            this.eventManager.post((Object)new DatabaseFreezeChangeEvent(true));
            this.freezeLocalDatabases();
        }
        return result;
    }

    @Override
    @Guarded(by={"STARTED"})
    public List<FreezeRequest> getState() {
        return Collections.unmodifiableList(this.databaseFrozenStateManager.getState());
    }

    @Override
    @Guarded(by={"STARTED"})
    public synchronized boolean releaseRequest(FreezeRequest request) {
        boolean result = this.databaseFrozenStateManager.remove(request);
        if (result && this.getState().isEmpty()) {
            this.releaseLocalDatabases();
            this.eventManager.post((Object)new DatabaseFreezeChangeEvent(false));
        }
        if (!result) {
            this.log.error("failed to release {}; freeze request state {}", (Object)request, this.getState());
        }
        return result;
    }

    @Override
    @Guarded(by={"STARTED"})
    public synchronized boolean releaseUserInitiatedIfPresent() {
        Optional<FreezeRequest> request = this.getState().stream().filter(it -> FreezeRequest.InitiatorType.USER_INITIATED.equals((Object)it.getInitiatorType())).findAny();
        if (request.isPresent()) {
            return this.releaseRequest(request.get());
        }
        return false;
    }

    @Override
    public synchronized List<FreezeRequest> releaseAllRequests() {
        List<FreezeRequest> requests = this.getState();
        for (FreezeRequest request : requests) {
            this.releaseRequest(request);
        }
        return requests;
    }

    @Override
    public synchronized void freezeLocalDatabases() {
        this.freezables.forEach(this::tryFreeze);
        if (this.nodeAccess.isClustered()) {
            this.setServerRole(ODistributedConfiguration.ROLES.REPLICA);
        } else {
            this.forEachFreezableDatabase(databaseInstance -> databaseInstance.setFrozen(true));
        }
    }

    @Override
    public synchronized void releaseLocalDatabases() {
        if (this.nodeAccess.isClustered()) {
            this.setServerRole(ODistributedConfiguration.ROLES.MASTER);
        } else {
            this.forEachFreezableDatabase(databaseInstance -> databaseInstance.setFrozen(false));
        }
        Lists.reverse(this.freezables).forEach(this::tryUnfreeze);
    }

    private void tryFreeze(Freezable freezable) {
        try {
            freezable.freeze();
        }
        catch (Exception e) {
            this.log.warn("Problem freezing {}", (Object)freezable, (Object)e);
        }
    }

    private void tryUnfreeze(Freezable freezable) {
        try {
            freezable.unfreeze();
        }
        catch (Exception e) {
            this.log.warn("Problem unfreezing {}", (Object)freezable, (Object)e);
        }
    }

    @Override
    public ReadOnlyState getReadOnlyState() {
        return new DefaultReadOnlyState(this.getState(), this.securityHelper.allPermitted(new Permission[]{READ_ONLY_PERMISSION}));
    }

    @Subscribe
    public void onNodeMerged(NodeMergedEvent event) {
        this.log.info("Node merged with existing cluster; shared frozen state is {}, local frozen state is={}", this.databaseFrozenStateManager.getState(), (Object)this.isFrozen());
        if (!this.isFrozen() && !this.getState().isEmpty()) {
            this.freezeLocalDatabases();
        } else if (this.isFrozen() && this.getState().isEmpty()) {
            this.releaseLocalDatabases();
        } else {
            this.log.info("no action taken for {}", (Object)event);
        }
    }

    private void forEachFreezableDatabase(Consumer<DatabaseInstance> databaseInstanceConsumer) {
        this.providers.forEach(provider -> {
            DatabaseInstance databaseInstance = (DatabaseInstance)provider.get();
            try {
                databaseInstanceConsumer.accept(databaseInstance);
            }
            catch (Exception e) {
                this.log.error("Unable to process Database instance: {}", (Object)databaseInstance, (Object)e);
            }
        });
    }

    private void setServerRole(ODistributedConfiguration.ROLES serverRole) {
        ODistributedServerManager distributedServerManager = (ODistributedServerManager)this.distributedServerManagerProvider.get();
        for (String database : distributedServerManager.getMessageService().getDatabases()) {
            this.log.info("Updating server role of {} database to {}", (Object)database, (Object)serverRole);
            distributedServerManager.executeInDistributedDatabaseLock(database, 0L, null, distributedConfiguration -> {
                distributedConfiguration.setServerRole(SERVER_NAME, serverRole);
                this.log.info("Updated server role of {} database to {}", (Object)database, (Object)serverRole);
                return null;
            });
        }
    }

    @VisibleForTesting
    void refreezeOnStartup(List<FreezeRequest> state) {
        this.log.info("Restoring database frozen state on startup");
        Map<FreezeRequest.InitiatorType, List<FreezeRequest>> requestsByInitiator = state.stream().collect(Collectors.groupingBy(FreezeRequest::getInitiatorType));
        for (FreezeRequest request : state) {
            this.log.warn("Database was frozen by {} process '{}' at {}", new Object[]{request.getInitiatorType(), request.getInitiatorId(), request.getTimestamp().withZone(DateTimeZone.getDefault())});
        }
        if (this.nodeAccess.isClustered()) {
            this.log.warn("Databases must be unfrozen manually");
            this.freezeLocalDatabases();
        } else {
            for (FreezeRequest request : requestsByInitiator.getOrDefault((Object)FreezeRequest.InitiatorType.SYSTEM, Collections.emptyList())) {
                this.log.warn("Discarding freeze request by {} process '{}'", (Object)request.getInitiatorType(), (Object)request.getInitiatorId());
                this.databaseFrozenStateManager.remove(request);
            }
            if (!requestsByInitiator.getOrDefault((Object)FreezeRequest.InitiatorType.USER_INITIATED, Collections.emptyList()).isEmpty()) {
                this.log.warn("Databases must be unfrozen manually");
                this.freezeLocalDatabases();
            }
        }
    }
}

