/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.repository.search.index;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.io.Resources;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Provider;
import javax.inject.Singleton;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.exists.indices.IndicesExistsResponse;
import org.elasticsearch.action.admin.indices.stats.IndexStats;
import org.elasticsearch.action.admin.indices.stats.IndicesStatsResponse;
import org.elasticsearch.action.bulk.BulkProcessor;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.IndicesAdminClient;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.indices.IndexAlreadyExistsException;
import org.sonatype.goodies.common.ComponentSupport;
import org.sonatype.nexus.common.event.EventManager;
import org.sonatype.nexus.repository.Repository;
import org.sonatype.nexus.repository.search.index.BulkIndexUpdateListener;
import org.sonatype.nexus.repository.search.index.BulkProcessorFlusher;
import org.sonatype.nexus.repository.search.index.BulkProcessorUpdater;
import org.sonatype.nexus.repository.search.index.ElasticSearchIndexService;
import org.sonatype.nexus.repository.search.index.IndexNamingPolicy;
import org.sonatype.nexus.repository.search.index.IndexSettingsContributor;
import org.sonatype.nexus.repository.search.index.JsonUtils;
import org.sonatype.nexus.scheduling.CancelableHelper;
import org.sonatype.nexus.thread.NexusThreadFactory;

@Named(value="default")
@Singleton
public class ElasticSearchIndexServiceImpl
extends ComponentSupport
implements ElasticSearchIndexService {
    private final Provider<Client> client;
    private final List<IndexSettingsContributor> indexSettingsContributors;
    private final EventManager eventManager;
    private final int calmTimeout;
    private final IndexNamingPolicy indexNamingPolicy;
    private final List<BulkIndexUpdateListener> updateListeners = new ArrayList<BulkIndexUpdateListener>();
    private final boolean periodicFlush;
    private final AtomicLong updateCount = new AtomicLong();
    private final ConcurrentMap<String, String> repositoryIndexNames = Maps.newConcurrentMap();
    private Map<Integer, Map.Entry<BulkProcessor, ExecutorService>> bulkProcessorToExecutors;

    @Inject
    public ElasticSearchIndexServiceImpl(Provider<Client> client, IndexNamingPolicy indexNamingPolicy, List<IndexSettingsContributor> indexSettingsContributors, EventManager eventManager, @Named(value="${nexus.elasticsearch.bulkCapacity:-1000}") int bulkCapacity, @Named(value="${nexus.elasticsearch.concurrentRequests:-1}") int concurrentRequests, @Named(value="${nexus.elasticsearch.flushInterval:-0}") int flushInterval, @Named(value="${nexus.elasticsearch.calmTimeout:-3000}") int calmTimeout, @Named(value="${nexus.elasticsearch.batching.threads.count:-1}") int batchingThreads) {
        Preconditions.checkState((batchingThreads > 0 ? 1 : 0) != 0, (Object)"'nexus.elasticsearch.batching.threads.count' must be positive.");
        this.client = (Provider)Preconditions.checkNotNull(client);
        this.indexNamingPolicy = (IndexNamingPolicy)Preconditions.checkNotNull((Object)indexNamingPolicy);
        this.indexSettingsContributors = (List)Preconditions.checkNotNull(indexSettingsContributors);
        this.eventManager = (EventManager)Preconditions.checkNotNull((Object)eventManager);
        this.calmTimeout = calmTimeout;
        this.periodicFlush = flushInterval > 0;
        this.createBulkProcessorsAndExecutors(bulkCapacity, concurrentRequests, flushInterval, batchingThreads);
    }

    private void createBulkProcessorsAndExecutors(int bulkCapacity, int concurrentRequests, int flushInterval, int batchingThreads) {
        HashMap<Integer, AbstractMap.SimpleImmutableEntry<BulkProcessor, ExecutorService>> bulkProcessorAndThreadPools = new HashMap<Integer, AbstractMap.SimpleImmutableEntry<BulkProcessor, ExecutorService>>();
        int count = 0;
        while (count < batchingThreads) {
            BulkIndexUpdateListener updateListener = new BulkIndexUpdateListener();
            this.updateListeners.add(updateListener);
            bulkProcessorAndThreadPools.put(count, new AbstractMap.SimpleImmutableEntry<BulkProcessor, ExecutorService>(BulkProcessor.builder((Client)((Client)this.client.get()), (BulkProcessor.Listener)updateListener).setBulkActions(bulkCapacity).setBulkSize(new ByteSizeValue(-1L)).setConcurrentRequests(concurrentRequests).setFlushInterval(this.periodicFlush ? TimeValue.timeValueMillis((long)flushInterval) : null).build(), this.createThreadPool(count)));
            ++count;
        }
        this.bulkProcessorToExecutors = Collections.unmodifiableMap(bulkProcessorAndThreadPools);
    }

    private ExecutorService createThreadPool(int id) {
        return Executors.newSingleThreadExecutor((ThreadFactory)new NexusThreadFactory("search-service-impl " + id, "search-service " + id));
    }

    @Override
    public void createIndex(Repository repository) {
        Preconditions.checkNotNull((Object)repository);
        String safeIndexName = this.indexNamingPolicy.indexName(repository);
        this.log.debug("Creating index for {}", (Object)repository);
        this.createIndex(repository, safeIndexName);
    }

    private void createIndex(Repository repository, String indexName) {
        IndicesAdminClient indices = this.indicesAdminClient();
        if (!((IndicesExistsResponse)indices.prepareExists(new String[]{indexName}).execute().actionGet()).isExists()) {
            ArrayList urls = Lists.newArrayListWithExpectedSize((int)(this.indexSettingsContributors.size() + 1));
            urls.add(Resources.getResource(this.getClass(), (String)"elasticsearch-mapping.json"));
            for (IndexSettingsContributor contributor : this.indexSettingsContributors) {
                URL url = contributor.getIndexSettings(repository);
                if (url == null) continue;
                urls.add(url);
            }
            try {
                String source = "{}";
                for (URL url : urls) {
                    this.log.debug("Merging ElasticSearch mapping: {}", (Object)url);
                    String contributed = Resources.toString((URL)url, (Charset)StandardCharsets.UTF_8);
                    this.log.trace("Contributed ElasticSearch mapping: {}", (Object)contributed);
                    source = JsonUtils.merge(source, contributed);
                }
                this.log.trace("ElasticSearch mapping: {}", (Object)source);
                indices.prepareCreate(indexName).setSource(source).execute().actionGet();
            }
            catch (IndexAlreadyExistsException e) {
                this.log.debug("Using existing index for {}", (Object)repository, (Object)e);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        this.repositoryIndexNames.put(repository.getName(), indexName);
    }

    @Override
    public void deleteIndex(Repository repository) {
        Preconditions.checkNotNull((Object)repository);
        String indexName = (String)this.repositoryIndexNames.remove(repository.getName());
        if (indexName != null) {
            this.log.debug("Removing index of {}", (Object)repository);
            this.deleteIndex(indexName);
        }
    }

    private void deleteIndex(String indexName) {
        this.flushBulkProcessors();
        IndicesAdminClient indices = this.indicesAdminClient();
        if (((IndicesExistsResponse)indices.prepareExists(new String[]{indexName}).execute().actionGet()).isExists()) {
            indices.prepareDelete(new String[]{indexName}).execute().actionGet();
        }
    }

    @Override
    public void rebuildIndex(Repository repository) {
        Preconditions.checkNotNull((Object)repository);
        String indexName = (String)this.repositoryIndexNames.remove(repository.getName());
        if (indexName != null) {
            this.log.debug("Rebuilding index for {}", (Object)repository);
            this.deleteIndex(indexName);
            this.createIndex(repository, indexName);
        }
    }

    @Override
    public boolean indexExist(Repository repository) {
        Preconditions.checkNotNull((Object)repository);
        String indexName = this.indexNamingPolicy.indexName(repository);
        IndicesAdminClient indices = this.indicesAdminClient();
        boolean indexExists = ((IndicesExistsResponse)indices.prepareExists(new String[]{indexName}).execute().actionGet()).isExists();
        this.log.info("Repository {} has search index: {}", (Object)repository, (Object)indexExists);
        return indexExists;
    }

    @Override
    public boolean indexEmpty(Repository repository) {
        Preconditions.checkNotNull((Object)repository);
        String indexName = this.indexNamingPolicy.indexName(repository);
        IndexStats indexStats = ((IndicesStatsResponse)this.indicesAdminClient().prepareStats(new String[]{indexName}).get()).getIndex(indexName);
        long count = 0L;
        if (indexStats != null) {
            count = indexStats.getTotal().getDocs().getCount();
        }
        boolean isEmpty = count == 0L;
        this.log.debug("Repository index: {} is {}.", (Object)indexName, (Object)(isEmpty ? "empty" : "not empty"));
        return isEmpty;
    }

    @Override
    public void put(Repository repository, final String identifier, String json) {
        Preconditions.checkNotNull((Object)repository);
        Preconditions.checkNotNull((Object)identifier);
        Preconditions.checkNotNull((Object)json);
        final String indexName = (String)this.repositoryIndexNames.get(repository.getName());
        if (indexName == null) {
            return;
        }
        this.updateCount.getAndIncrement();
        this.log.debug("Adding to index document {} from {}: {}", new Object[]{identifier, repository, json});
        ((Client)this.client.get()).prepareIndex(indexName, "component", identifier).setSource(json).execute((ActionListener)new ActionListener<IndexResponse>(){

            public void onResponse(IndexResponse indexResponse) {
                ElasticSearchIndexServiceImpl.this.log.debug("successfully added {} {} to index {}: {}", new Object[]{"component", identifier, indexName, indexResponse});
            }

            public void onFailure(Throwable e) {
                ElasticSearchIndexServiceImpl.this.log.error("failed to add {} {} to index {}; this is a sign that the Elasticsearch index thread pool is overloaded", new Object[]{"component", identifier, indexName, e});
            }
        });
    }

    @Override
    public <T> List<Future<Void>> bulkPut(Repository repository, Iterable<T> components, Function<T, String> identifierProducer, Function<T, String> jsonDocumentProducer) {
        Preconditions.checkNotNull((Object)repository);
        Preconditions.checkNotNull(components);
        String indexName = (String)this.repositoryIndexNames.get(repository.getName());
        if (indexName == null) {
            return Collections.emptyList();
        }
        Map.Entry<BulkProcessor, ExecutorService> bulkProcessorToExecutorPair = this.pickABulkProcessor();
        BulkProcessor bulkProcessor = bulkProcessorToExecutorPair.getKey();
        ExecutorService executorService = bulkProcessorToExecutorPair.getValue();
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>();
        components.forEach(component -> {
            CancelableHelper.checkCancellation();
            String identifier = (String)identifierProducer.apply(component);
            String json = (String)jsonDocumentProducer.apply(component);
            if (json != null) {
                this.updateCount.getAndIncrement();
                this.log.debug("Bulk adding to index document {} from {}: {}", new Object[]{identifier, repository, json});
                futures.add(executorService.submit(new BulkProcessorUpdater<IndexRequest>(bulkProcessor, this.createIndexRequest(indexName, identifier, json))));
            }
        });
        if (!this.periodicFlush) {
            futures.add(executorService.submit(new BulkProcessorFlusher(bulkProcessor)));
        }
        return futures;
    }

    private IndexRequest createIndexRequest(String indexName, String identifier, String json) {
        return (IndexRequest)((Client)this.client.get()).prepareIndex(indexName, "component", identifier).setSource(json).request();
    }

    @Override
    public void delete(Repository repository, final String identifier) {
        Preconditions.checkNotNull((Object)repository);
        Preconditions.checkNotNull((Object)identifier);
        final String indexName = (String)this.repositoryIndexNames.get(repository.getName());
        if (indexName == null) {
            return;
        }
        this.log.debug("Removing from index document {} from {}", (Object)identifier, (Object)repository);
        ((Client)this.client.get()).prepareDelete(indexName, "component", identifier).execute((ActionListener)new ActionListener<DeleteResponse>(){

            public void onResponse(DeleteResponse deleteResponse) {
                ElasticSearchIndexServiceImpl.this.log.debug("successfully removed {} {} from index {}: {}", new Object[]{"component", identifier, indexName, deleteResponse});
            }

            public void onFailure(Throwable e) {
                ElasticSearchIndexServiceImpl.this.log.error("failed to remove {} {} from index {}; this is a sign that the Elasticsearch index thread pool is overloaded", new Object[]{"component", identifier, indexName, e});
            }
        });
    }

    @Override
    public void bulkDelete(@Nullable Repository repository, Iterable<String> identifiers) {
        Preconditions.checkNotNull(identifiers);
        Map.Entry<BulkProcessor, ExecutorService> bulkProcessorToExecutorPair = this.pickABulkProcessor();
        BulkProcessor bulkProcessor = bulkProcessorToExecutorPair.getKey();
        ExecutorService executorService = bulkProcessorToExecutorPair.getValue();
        if (repository != null) {
            String indexName = (String)this.repositoryIndexNames.get(repository.getName());
            if (indexName == null) {
                return;
            }
            identifiers.forEach(id -> {
                this.log.debug("Bulk removing from index document {} from {}", id, (Object)repository);
                DeleteRequest deleteRequest = (DeleteRequest)((Client)this.client.get()).prepareDelete(indexName, "component", id).request();
                executorService.submit(new BulkProcessorUpdater<DeleteRequest>(bulkProcessor, deleteRequest));
            });
        } else {
            Iterables.partition(identifiers, (int)100).forEach(chunk -> {
                SearchResponse toDelete = (SearchResponse)((Client)this.client.get()).prepareSearch(new String[]{"_all"}).setFetchSource(false).setQuery((QueryBuilder)QueryBuilders.idsQuery((String[])new String[]{"component"}).ids((Collection)chunk)).setSize(chunk.size()).execute().actionGet();
                toDelete.getHits().forEach(hit -> {
                    this.log.debug("Bulk removing from index document {} from {}", (Object)hit.getId(), (Object)hit.index());
                    DeleteRequest request = (DeleteRequest)((Client)this.client.get()).prepareDelete(hit.index(), "component", hit.getId()).request();
                    executorService.submit(new BulkProcessorUpdater<DeleteRequest>(bulkProcessor, request));
                });
            });
        }
        if (!this.periodicFlush) {
            executorService.submit(new BulkProcessorFlusher(bulkProcessor));
        }
    }

    private Map.Entry<BulkProcessor, ExecutorService> pickABulkProcessor() {
        int numberOfBulkProcessors = this.bulkProcessorToExecutors.size();
        if (numberOfBulkProcessors > 1) {
            int index = ThreadLocalRandom.current().nextInt(numberOfBulkProcessors);
            return this.bulkProcessorToExecutors.get(index);
        }
        return this.bulkProcessorToExecutors.get(0);
    }

    @Override
    public void flush(boolean fsync) {
        this.log.debug("Flushing index requests");
        this.flushBulkProcessors();
        if (fsync) {
            try {
                this.indicesAdminClient().prepareSyncedFlush(new String[0]).execute().actionGet();
            }
            catch (RuntimeException e) {
                this.log.warn("Problem flushing search indices", (Throwable)e);
            }
        }
    }

    private List<Future<Void>> flushBulkProcessors() {
        return this.bulkProcessorToExecutors.values().stream().map(this::flushBulkProcessor).collect(Collectors.toList());
    }

    private Future<Void> flushBulkProcessor(Map.Entry<BulkProcessor, ExecutorService> bulkProcessorExecutorPair) {
        ExecutorService executorService = bulkProcessorExecutorPair.getValue();
        BulkProcessor bulkProcessor = bulkProcessorExecutorPair.getKey();
        return executorService.submit(new BulkProcessorFlusher(bulkProcessor));
    }

    @Override
    public long getUpdateCount() {
        return this.updateCount.get();
    }

    @Override
    public boolean isCalmPeriod() {
        if (this.isUpdateInFlight()) {
            return false;
        }
        try {
            this.indicesAdminClient().prepareRefresh(new String[0]).execute().actionGet();
        }
        catch (RuntimeException e) {
            this.log.warn("Problem refreshing search indices", (Throwable)e);
        }
        return true;
    }

    private boolean isUpdateInFlight() {
        return this.updateListeners.stream().map(BulkIndexUpdateListener::inflightRequestCount).anyMatch(inflight -> inflight > 0);
    }

    @Override
    public void waitForCalm() {
        try {
            this.waitFor(() -> ((EventManager)this.eventManager).isCalmPeriod());
            this.flush(false);
            this.waitFor(this::isCalmPeriod);
        }
        catch (InterruptedException e) {
            throw new RuntimeException("Waiting for calm period has been interrupted", e);
        }
    }

    private void waitFor(Callable<Boolean> function) throws InterruptedException {
        Thread.yield();
        long end = System.currentTimeMillis() + (long)this.calmTimeout;
        do {
            try {
                if (Boolean.TRUE.equals(function.call())) {
                    return;
                }
            }
            catch (InterruptedException e) {
                throw e;
            }
            catch (Exception e) {
                this.log.debug("Exception thrown whilst waiting", (Throwable)e);
            }
            Thread.sleep(100L);
        } while (System.currentTimeMillis() <= end);
        this.log.warn("Timed out waiting for {} after {} ms", function, (Object)this.calmTimeout);
    }

    private IndicesAdminClient indicesAdminClient() {
        return ((Client)this.client.get()).admin().indices();
    }
}

