/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.ssl.internal.geronimo;

import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SignatureException;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.annotation.concurrent.NotThreadSafe;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonatype.nexus.crypto.CryptoHelper;
import org.sonatype.nexus.ssl.CertificateUtil;
import org.sonatype.nexus.ssl.KeyNotFoundException;
import org.sonatype.nexus.ssl.KeystoreException;
import org.sonatype.nexus.ssl.internal.geronimo.KeystoreInstance;
import org.sonatype.nexus.ssl.spi.KeyStoreStorage;

@NotThreadSafe
public class FileKeystoreInstance
implements KeystoreInstance {
    private static final Logger log = LoggerFactory.getLogger(FileKeystoreInstance.class);
    private final CryptoHelper crypto;
    private KeyStoreStorage storage;
    private String keystoreName;
    private String keystoreType;
    private char[] keystorePassword;
    private Map<String, char[]> keyPasswords = new HashMap<String, char[]>();
    private char[] openPassword;
    private List<String> privateKeys = new ArrayList<String>();
    private List<String> trustCerts = new ArrayList<String>();
    private KeyStore keystore;

    public FileKeystoreInstance(CryptoHelper crypto, KeyStoreStorage storage, String keystoreName, char[] keystorePassword, String keystoreType, Map<String, char[]> keyPasswords) {
        this.crypto = crypto;
        this.storage = storage;
        this.keystoreName = keystoreName;
        this.keystoreType = keystoreType;
        this.keystorePassword = keystorePassword;
        if (keyPasswords != null) {
            this.keyPasswords.putAll(keyPasswords);
        }
        this.initializeKeystoreIfNotExist();
    }

    public String toString() {
        return String.valueOf(this.getClass().getSimpleName()) + "{" + "storage=" + this.storage + ", keystoreType='" + this.keystoreType + '\'' + '}';
    }

    private void initializeKeystoreIfNotExist() {
        if (!this.storage.exists()) {
            log.debug("keystore does not exist, creating new one of type: {}", (Object)this.keystoreType);
            try {
                KeyStore tempKeystore = this.crypto.createKeyStore(this.keystoreType);
                tempKeystore.load(null, this.keystorePassword);
                this.keystore = tempKeystore;
                this.saveKeystore(this.keystorePassword);
                this.loadKeystoreData(this.keystorePassword);
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Invalid keystore storage (" + this.storage + ")", e);
            }
        }
    }

    @Override
    public String[] listPrivateKeys(char[] storePassword) throws KeystoreException {
        this.ensureLoaded(storePassword);
        return this.privateKeys.toArray(new String[this.privateKeys.size()]);
    }

    @Override
    public String[] listTrustCertificates(char[] storePassword) throws KeystoreException {
        this.ensureLoaded(storePassword);
        return this.trustCerts.toArray(new String[this.trustCerts.size()]);
    }

    @Override
    public void importTrustCertificate(Certificate cert, String alias, char[] storePassword) throws KeystoreException {
        if (storePassword == null) {
            throw new NullPointerException("storePassword is null");
        }
        this.ensureLoaded(storePassword);
        try {
            this.keystore.setCertificateEntry(alias, cert);
        }
        catch (KeyStoreException e) {
            throw new KeystoreException("Unable to set certificate entry in keystore '" + this.keystoreName + "' for alias '" + alias + "'", e);
        }
        this.trustCerts.add(alias);
        this.saveKeystore(storePassword);
    }

    @Override
    public void generateKeyPair(String alias, char[] storePassword, char[] keyPassword, String keyAlgorithm, int keySize, String signatureAlgorithm, int validity, String commonName, String orgUnit, String organization, String locality, String state, String country) throws KeystoreException {
        if (storePassword == null) {
            throw new NullPointerException("storePassword is null");
        }
        this.ensureLoaded(storePassword);
        try {
            KeyPairGenerator kpgen = this.crypto.createKeyPairGenerator(keyAlgorithm);
            kpgen.initialize(keySize);
            KeyPair keyPair = kpgen.generateKeyPair();
            X509Certificate cert = this.generateCertificate(keyPair.getPublic(), keyPair.getPrivate(), signatureAlgorithm, validity, commonName, orgUnit, organization, locality, state, country);
            this.keystore.setKeyEntry(alias, keyPair.getPrivate(), keyPassword, new Certificate[]{cert});
            this.privateKeys.add(alias);
            this.keyPasswords.put(alias, keyPassword);
        }
        catch (KeyStoreException e) {
            throw new KeystoreException("Unable to generate key pair in keystore '" + this.keystoreName + "'", e);
        }
        catch (InvalidKeyException e) {
            throw new KeystoreException("Unable to generate key pair in keystore '" + this.keystoreName + "'", e);
        }
        catch (SignatureException e) {
            throw new KeystoreException("Unable to generate key pair in keystore '" + this.keystoreName + "'", e);
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeystoreException("Unable to generate key pair in keystore '" + this.keystoreName + "'", e);
        }
        catch (CertificateEncodingException e) {
            throw new KeystoreException("Unable to generate key pair in keystore '" + this.keystoreName + "'", e);
        }
        this.saveKeystore(storePassword);
    }

    @Override
    public void deleteEntry(String alias, char[] storePassword) throws KeystoreException {
        if (storePassword == null) {
            throw new NullPointerException("storePassword is null");
        }
        this.ensureLoaded(storePassword);
        try {
            this.keystore.deleteEntry(alias);
        }
        catch (KeyStoreException e) {
            throw new KeystoreException("Unable to delete key in keystore '" + this.keystoreName + "' for alias '" + alias + "'", e);
        }
        this.privateKeys.remove(alias);
        this.trustCerts.remove(alias);
        if (this.keyPasswords.containsKey(alias)) {
            this.keyPasswords.remove(alias);
        }
        this.saveKeystore(storePassword);
    }

    @Override
    public KeyManager[] getKeyManager(String algorithm, String alias, char[] storePassword) throws KeystoreException {
        this.ensureLoaded(storePassword);
        try {
            KeyManagerFactory keyFactory = this.crypto.createKeyManagerFactory(algorithm);
            if (this.privateKeys.size() == 1) {
                keyFactory.init(this.keystore, this.keyPasswords.get(alias));
            } else {
                KeyStore subKeystore = KeyStore.getInstance(this.keystore.getType(), this.keystore.getProvider());
                try {
                    subKeystore.load(null, null);
                }
                catch (NoSuchAlgorithmException noSuchAlgorithmException) {
                }
                catch (CertificateException certificateException) {
                }
                catch (IOException iOException) {}
                subKeystore.setKeyEntry(alias, this.keystore.getKey(alias, this.keyPasswords.get(alias)), this.keyPasswords.get(alias), this.keystore.getCertificateChain(alias));
                keyFactory.init(subKeystore, this.keyPasswords.get(alias));
            }
            return keyFactory.getKeyManagers();
        }
        catch (KeyStoreException e) {
            throw new KeystoreException("Unable to retrieve key manager in keystore '" + this.keystoreName + "' for alias '" + alias + "'", e);
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeystoreException("Unable to retrieve key manager in keystore '" + this.keystoreName + "' for alias '" + alias + "'", e);
        }
        catch (UnrecoverableKeyException e) {
            throw new KeystoreException("Unable to retrieve key manager in keystore '" + this.keystoreName + "' for alias '" + alias + "'", e);
        }
    }

    @Override
    public TrustManager[] getTrustManager(String algorithm, char[] storePassword) throws KeystoreException {
        this.ensureLoaded(storePassword);
        try {
            TrustManagerFactory trustFactory = this.crypto.createTrustManagerFactory(algorithm);
            trustFactory.init(this.keystore);
            return trustFactory.getTrustManagers();
        }
        catch (KeyStoreException e) {
            throw new KeystoreException("Unable to retrieve trust manager in keystore '" + this.keystoreName + "'", e);
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeystoreException("Unable to retrieve trust manager in keystore '" + this.keystoreName + "'", e);
        }
    }

    @Override
    public PrivateKey getPrivateKey(String alias, char[] storePassword, char[] keyPassword) throws KeystoreException {
        this.ensureLoaded(storePassword);
        try {
            PrivateKey key = (PrivateKey)this.keystore.getKey(alias, keyPassword);
            if (key == null) {
                throw new KeyNotFoundException("Keystore '" + this.keystoreName + "' does not contain a private key with alias '" + alias + "'.");
            }
            return key;
        }
        catch (KeyStoreException e) {
            throw new KeystoreException("Unable to retrieve private key from keystore", e);
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeystoreException("Unable to retrieve private key from keystore", e);
        }
        catch (UnrecoverableKeyException e) {
            throw new KeystoreException("Unable to retrieve private key from keystore", e);
        }
    }

    @Override
    public Certificate getCertificate(String alias, char[] storePassword) throws KeystoreException {
        this.ensureLoaded(storePassword);
        try {
            Certificate cert = this.keystore.getCertificate(alias);
            if (cert == null) {
                throw new KeyNotFoundException("Keystore '" + this.keystoreName + "' does not contain a certificate with alias '" + alias + "'.");
            }
            return cert;
        }
        catch (KeyStoreException e) {
            throw new KeystoreException("Unable to retrieve certificate from keystore", e);
        }
    }

    @Override
    public Certificate getCertificate(String alias) {
        try {
            return this.keystore.getCertificate(alias);
        }
        catch (KeyStoreException e) {
            log.error("Unable to read certificate from keystore", (Throwable)e);
            return null;
        }
    }

    private void loadKeystoreData(char[] password) throws KeystoreException {
        try {
            KeyStore tempKeystore = this.crypto.createKeyStore(this.keystoreType);
            this.storage.load(tempKeystore, password);
            this.keystore = tempKeystore;
            this.privateKeys.clear();
            this.trustCerts.clear();
            this.openPassword = password;
            Enumeration<String> aliases = this.keystore.aliases();
            while (aliases.hasMoreElements()) {
                String alias = aliases.nextElement();
                if (this.keystore.isKeyEntry(alias)) {
                    this.privateKeys.add(alias);
                    continue;
                }
                if (!this.keystore.isCertificateEntry(alias)) continue;
                this.trustCerts.add(alias);
            }
        }
        catch (KeyStoreException e) {
            throw new KeystoreException("Unable to open keystore with provided password", e);
        }
        catch (IOException e) {
            throw new KeystoreException("Unable to open keystore with provided password", e);
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeystoreException("Unable to open keystore with provided password", e);
        }
        catch (CertificateException e) {
            throw new KeystoreException("Unable to open keystore with provided password", e);
        }
    }

    private boolean isLoaded(char[] password) {
        if (this.openPassword == null || this.openPassword.length != password.length) {
            return false;
        }
        if (this.storage.modified()) {
            return false;
        }
        int i = 0;
        while (i < password.length) {
            if (password[i] != this.openPassword[i]) {
                return false;
            }
            ++i;
        }
        return true;
    }

    private void ensureLoaded(char[] storePassword) throws KeystoreException {
        char[] password = storePassword == null ? this.keystorePassword : storePassword;
        if (!this.isLoaded(password)) {
            this.loadKeystoreData(password);
        }
    }

    private X509Certificate generateCertificate(PublicKey publicKey, PrivateKey privateKey, String algorithm, int validity, String commonName, String orgUnit, String organization, String locality, String state, String country) throws SignatureException, InvalidKeyException, NoSuchAlgorithmException, CertificateEncodingException {
        return CertificateUtil.generateCertificate(publicKey, privateKey, algorithm, validity, commonName, orgUnit, organization, locality, state, country);
    }

    private void saveKeystore(char[] password) throws KeystoreException {
        try {
            this.storage.save(this.keystore, password);
        }
        catch (KeyStoreException e) {
            throw new KeystoreException("Unable to save keystore '" + this.keystoreName + "'", e);
        }
        catch (IOException e) {
            throw new KeystoreException("Unable to save keystore '" + this.keystoreName + "'", e);
        }
        catch (NoSuchAlgorithmException e) {
            throw new KeystoreException("Unable to save keystore '" + this.keystoreName + "'", e);
        }
        catch (CertificateException e) {
            throw new KeystoreException("Unable to save keystore '" + this.keystoreName + "'", e);
        }
    }
}

