/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.blobstore;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.LocalDate;
import java.time.format.DateTimeParseException;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.stream.Stream;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.sonatype.nexus.blobstore.api.BlobId;
import org.sonatype.nexus.common.app.ApplicationDirectories;

@Singleton
public class BlobStoreReconciliationLogger {
    public static final String BLOBSTORE = "blobstore-reconciliation-path";
    private static final String RECONCILIATION_LOGGER_NAME = "blobstore-reconciliation-log";
    private static final Logger LOGGER = LoggerFactory.getLogger(BlobStoreReconciliationLogger.class);
    private final Logger reconciliationLogger;
    private final ApplicationDirectories applicationDirectories;

    @Inject
    public BlobStoreReconciliationLogger(ApplicationDirectories applicationDirectories) {
        this.applicationDirectories = (ApplicationDirectories)Preconditions.checkNotNull((Object)applicationDirectories);
        this.reconciliationLogger = LoggerFactory.getLogger((String)RECONCILIATION_LOGGER_NAME);
    }

    public void logBlobCreated(Path reconciliationLogPath, BlobId blobId) {
        if (this.isNotTemporaryBlob(blobId)) {
            MDC.put((String)BLOBSTORE, (String)reconciliationLogPath.toString());
            this.reconciliationLogger.info(blobId.asUniqueString());
            MDC.remove((String)BLOBSTORE);
        }
    }

    private boolean isNotTemporaryBlob(BlobId blobId) {
        return !blobId.asUniqueString().startsWith("tmp$");
    }

    public Stream<BlobId> getBlobsCreatedSince(Path reconciliationLogPath, LocalDate sinceDate) {
        return this.getLogFilesToProcess(reconciliationLogPath, sinceDate).flatMap(this::readLines).map(line -> {
            String[] split = line.split(",");
            if (split.length == 2) {
                return split[1];
            }
            LOGGER.info("Cannot find blob id on line, skipping: {}", line);
            return null;
        }).filter(Objects::nonNull).map(BlobId::new);
    }

    private Stream<String> readLines(File file) {
        try {
            return Files.lines(file.toPath());
        }
        catch (IOException e) {
            LOGGER.error("Problem when reading file '{}'", (Object)file.getName(), (Object)e);
            return Stream.empty();
        }
    }

    private Stream<File> getLogFilesToProcess(Path reconciliationLogPath, LocalDate sinceDate) {
        File reconciliationLogDirectory = this.applicationDirectories.getWorkDirectory(reconciliationLogPath.toString());
        File[] logs = reconciliationLogDirectory.listFiles();
        if (Objects.nonNull(logs)) {
            return Stream.of(logs).filter(this.isFileNameOlderOrSameAs(sinceDate)).peek(file -> LOGGER.info("Processing file '{}'", (Object)file.getName()));
        }
        LOGGER.info("No files found to process");
        return Stream.empty();
    }

    private Predicate<File> isFileNameOlderOrSameAs(LocalDate sinceDate) {
        return file -> {
            try {
                LocalDate logFileDate = LocalDate.parse(file.getName());
                return !sinceDate.isAfter(logFileDate);
            }
            catch (DateTimeParseException dateTimeParseException) {
                return false;
            }
        };
    }
}

