/*
 * Decompiled with CFR 0.152.
 */
package zz.com.github.packageurl;

import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import zz.com.github.packageurl.MalformedPackageURLException;
import zz.com.github.packageurl.ValidationException;

public final class PackageURL
implements Serializable {
    private static final long serialVersionUID = 3243226021636427586L;
    private static final String UTF8 = StandardCharsets.UTF_8.name();
    private static final Pattern PATH_SPLITTER = Pattern.compile("/");
    private String scheme;
    private String type;
    private String namespace;
    private String name;
    private String version;
    private Map<String, String> qualifiers;
    private String subpath;
    private String canonicalizedForm = null;

    public PackageURL(String purl) throws MalformedPackageURLException {
        this.parse(purl);
    }

    public PackageURL(String type, String name) throws MalformedPackageURLException {
        this(type, null, name, null, null, null);
    }

    public PackageURL(String type, String namespace, String name, String version, TreeMap<String, String> qualifiers, String subpath) throws MalformedPackageURLException {
        this.scheme = this.validateScheme("pkg");
        this.type = this.validateType(type);
        this.namespace = this.validateNamespace(namespace);
        this.name = this.validateName(name);
        this.version = this.validateVersion(version);
        this.qualifiers = this.validateQualifiers(qualifiers);
        this.subpath = this.validatePath(subpath, true);
        this.verifyTypeConstraints(this.type, this.namespace, this.name);
    }

    public String getScheme() {
        return this.scheme;
    }

    public String getType() {
        return this.type;
    }

    public String getNamespace() {
        return this.namespace;
    }

    public String getName() {
        return this.name;
    }

    public String getVersion() {
        return this.version;
    }

    public Map<String, String> getQualifiers() {
        return this.qualifiers;
    }

    public String getSubpath() {
        return this.subpath;
    }

    private String validateScheme(String value) throws MalformedPackageURLException {
        if ("pkg".equals(value)) {
            return "pkg";
        }
        throw new MalformedPackageURLException("The PackageURL scheme is invalid");
    }

    private String validateType(String value) throws MalformedPackageURLException {
        if (value == null || value.isEmpty()) {
            throw new MalformedPackageURLException("The PackageURL type cannot be null or empty");
        }
        if (value.charAt(0) >= '0' && value.charAt(0) <= '9') {
            throw new MalformedPackageURLException("The PackageURL type cannot start with a number");
        }
        String retVal = value.toLowerCase();
        if (retVal.chars().anyMatch(c -> !(c == 46 || c == 43 || c == 45 || c >= 97 && c <= 122 || c >= 48 && c <= 57))) {
            throw new MalformedPackageURLException("The PackageURL type contains invalid characters");
        }
        return retVal;
    }

    private String validateNamespace(String value) throws MalformedPackageURLException {
        if (value == null || value.isEmpty()) {
            return null;
        }
        return this.validateNamespace(value.split("/"));
    }

    private String validateNamespace(String[] values) throws MalformedPackageURLException {
        String retVal;
        if (values == null || values.length == 0) {
            return null;
        }
        String tempNamespace = this.validatePath(values, false);
        switch (this.type) {
            case "bitbucket": 
            case "deb": 
            case "github": 
            case "golang": 
            case "rpm": {
                retVal = tempNamespace.toLowerCase();
                break;
            }
            default: {
                retVal = tempNamespace;
            }
        }
        return retVal;
    }

    private String validateName(String value) throws MalformedPackageURLException {
        String temp;
        if (value == null || value.isEmpty()) {
            throw new MalformedPackageURLException("The PackageURL name specified is invalid");
        }
        switch (this.type) {
            case "bitbucket": 
            case "deb": 
            case "github": 
            case "golang": {
                temp = value.toLowerCase();
                break;
            }
            case "pypi": {
                temp = value.replaceAll("_", "-").toLowerCase();
                break;
            }
            default: {
                temp = value;
            }
        }
        return temp;
    }

    private String validateVersion(String value) {
        if (value == null) {
            return null;
        }
        return value;
    }

    private Map<String, String> validateQualifiers(Map<String, String> values) throws MalformedPackageURLException {
        if (values == null) {
            return null;
        }
        for (Map.Entry<String, String> entry : values.entrySet()) {
            this.validateKey(entry.getKey());
            String value = entry.getValue();
            if (value != null && !value.isEmpty()) continue;
            throw new MalformedPackageURLException("The PackageURL specified contains a qualifier key with an empty or null value");
        }
        return values;
    }

    private String validateKey(String value) throws MalformedPackageURLException {
        if (value == null || value.isEmpty()) {
            throw new MalformedPackageURLException("Qualifier key is invalid: " + value);
        }
        String retValue = value.toLowerCase();
        if (value.charAt(0) >= '0' && value.charAt(0) <= '9' || !value.chars().allMatch(c -> c >= 97 && c <= 122 || c >= 48 && c <= 57 || c == 46 || c == 45 || c == 95)) {
            throw new MalformedPackageURLException("Qualifier key is invalid: " + value);
        }
        return retValue;
    }

    private String validatePath(String value, boolean isSubpath) throws MalformedPackageURLException {
        if (value == null || value.isEmpty()) {
            return null;
        }
        return this.validatePath(value.split("/"), isSubpath);
    }

    private String validatePath(String[] segments, boolean isSubpath) throws MalformedPackageURLException {
        if (segments == null || segments.length == 0) {
            return null;
        }
        try {
            return Arrays.stream(segments).map(segment -> {
                if (isSubpath && ("..".equals(segment) || ".".equals(segment))) {
                    throw new ValidationException("Segments in the subpath may not be a period ('.') or repeated period ('..')");
                }
                if (segment.contains("/")) {
                    throw new ValidationException("Segments in the namespace and subpath may not contain a forward slash ('/')");
                }
                if (segment.isEmpty()) {
                    throw new ValidationException("Segments in the namespace and subpath may not be empty");
                }
                return segment;
            }).collect(Collectors.joining("/"));
        }
        catch (ValidationException ex) {
            throw new MalformedPackageURLException(ex.getMessage());
        }
    }

    public String toString() {
        return this.canonicalize();
    }

    public String canonicalize() {
        return this.canonicalize(false);
    }

    private String canonicalize(boolean coordinatesOnly) {
        if (this.canonicalizedForm != null) {
            return this.canonicalizedForm;
        }
        StringBuilder purl = new StringBuilder();
        purl.append(this.scheme).append(":");
        if (this.type != null) {
            purl.append(this.type);
        }
        purl.append("/");
        if (this.namespace != null) {
            purl.append(this.encodePath(this.namespace));
            purl.append("/");
        }
        if (this.name != null) {
            purl.append(this.percentEncode(this.name));
        }
        if (this.version != null) {
            purl.append("@").append(this.percentEncode(this.version));
        }
        if (!coordinatesOnly) {
            if (this.qualifiers != null && this.qualifiers.size() > 0) {
                purl.append("?");
                this.qualifiers.entrySet().stream().forEachOrdered(entry -> {
                    purl.append(((String)entry.getKey()).toLowerCase());
                    purl.append("=");
                    purl.append(this.percentEncode((String)entry.getValue()));
                    purl.append("&");
                });
                purl.setLength(purl.length() - 1);
            }
            if (this.subpath != null) {
                purl.append("#").append(this.encodePath(this.subpath));
            }
        }
        this.canonicalizedForm = purl.toString();
        return this.canonicalizedForm;
    }

    private String percentEncode(String input) {
        try {
            return URLEncoder.encode(input, UTF8).replace("+", "%20").replace("*", "%2A").replace("%7E", "~");
        }
        catch (UnsupportedEncodingException e) {
            return input;
        }
    }

    private String percentDecode(String input) {
        if (input == null) {
            return null;
        }
        try {
            String decoded = URLDecoder.decode(input, UTF8);
            if (!decoded.equals(input)) {
                return decoded;
            }
        }
        catch (UnsupportedEncodingException e) {
            return input;
        }
        return input;
    }

    private void parse(String purl) throws MalformedPackageURLException {
        if (purl == null || purl.trim().isEmpty()) {
            throw new MalformedPackageURLException("Invalid purl: Contains an empty or null value");
        }
        try {
            int start;
            int end;
            StringBuilder remainder;
            int index;
            URI uri = new URI(purl);
            if (uri.getUserInfo() != null || uri.getPort() != -1) {
                throw new MalformedPackageURLException("Invalid purl: Contains parts not supported by the purl spec");
            }
            this.scheme = this.validateScheme(uri.getScheme());
            if (uri.getRawFragment() != null && !uri.getRawFragment().isEmpty()) {
                this.subpath = this.validatePath(this.parsePath(uri.getRawFragment(), true), true);
            }
            if ((index = (remainder = new StringBuilder(uri.getRawSchemeSpecificPart())).lastIndexOf("?")) >= 0) {
                this.qualifiers = this.parseQualifiers(remainder.substring(index + 1));
                remainder.setLength(index);
            }
            for (end = remainder.length() - 1; end > 0 && '/' == remainder.charAt(end); --end) {
            }
            if (end < remainder.length() - 1) {
                remainder.setLength(end + 1);
            }
            for (start = 0; start < remainder.length() && '/' == remainder.charAt(start); ++start) {
            }
            index = remainder.indexOf("/", start);
            if (index <= start) {
                throw new MalformedPackageURLException("Invalid purl: does not contain both a type and name");
            }
            this.type = this.validateType(remainder.substring(start, index).toLowerCase());
            start = index + 1;
            index = remainder.lastIndexOf("@");
            if (index >= start) {
                this.version = this.validateVersion(this.percentDecode(remainder.substring(index + 1)));
                remainder.setLength(index);
            }
            if ((index = remainder.lastIndexOf("/")) <= start) {
                this.name = this.validateName(this.percentDecode(remainder.substring(start)));
            } else {
                this.name = this.validateName(this.percentDecode(remainder.substring(index + 1)));
                remainder.setLength(index);
                this.namespace = this.validateNamespace(this.parsePath(remainder.substring(start), false));
            }
            this.verifyTypeConstraints(this.type, this.namespace, this.name);
        }
        catch (URISyntaxException e) {
            throw new MalformedPackageURLException("Invalid purl: " + e.getMessage());
        }
    }

    private void verifyTypeConstraints(String type, String namespace, String name) throws MalformedPackageURLException {
        if ("maven".equals(type) && (namespace == null || namespace.isEmpty() || name == null || name.isEmpty())) {
            throw new MalformedPackageURLException("The PackageURL specified is invalid. Maven requires both a namespace and name.");
        }
    }

    private Map<String, String> parseQualifiers(String encodedString) throws MalformedPackageURLException {
        try {
            TreeMap results = Arrays.stream(encodedString.split("&")).collect(TreeMap::new, (map, value) -> {
                String[] entry = value.split("=", 2);
                if (entry.length == 2 && !entry[1].isEmpty() && map.put(entry[0].toLowerCase(), this.percentDecode(entry[1])) != null) {
                    throw new ValidationException("Duplicate package qualifier encountere - more then one value was specified for " + entry[0].toLowerCase());
                }
            }, TreeMap::putAll);
            return this.validateQualifiers(results);
        }
        catch (ValidationException ex) {
            throw new MalformedPackageURLException(ex.getMessage());
        }
    }

    private String[] parsePath(String value, boolean isSubpath) throws MalformedPackageURLException {
        if (value == null || value.isEmpty()) {
            return null;
        }
        return (String[])PATH_SPLITTER.splitAsStream(value).filter(segment -> !segment.isEmpty() && (!isSubpath || !".".equals(segment) && !"..".equals(segment))).map(segment -> this.percentDecode((String)segment)).toArray(String[]::new);
    }

    private String encodePath(String path) {
        return Arrays.stream(path.split("/")).map(segment -> this.percentEncode((String)segment)).collect(Collectors.joining("/"));
    }

    @Deprecated
    public boolean isBaseEquals(PackageURL purl) {
        return this.isCoordinatesEquals(purl);
    }

    public boolean isCoordinatesEquals(PackageURL purl) {
        return Objects.equals(this.scheme, purl.scheme) && Objects.equals(this.type, purl.type) && Objects.equals(this.namespace, purl.namespace) && Objects.equals(this.name, purl.name) && Objects.equals(this.version, purl.version);
    }

    public String getCoordinates() {
        return this.canonicalize(true);
    }

    public boolean isCanonicalEquals(PackageURL purl) {
        return this.canonicalize().equals(purl.canonicalize());
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        PackageURL other = (PackageURL)o;
        return Objects.equals(this.scheme, other.scheme) && Objects.equals(this.type, other.type) && Objects.equals(this.namespace, other.namespace) && Objects.equals(this.name, other.name) && Objects.equals(this.version, other.version) && Objects.equals(this.qualifiers, other.qualifiers) && Objects.equals(this.subpath, other.subpath);
    }

    public int hashCode() {
        return Objects.hash(this.scheme, this.type, this.namespace, this.name, this.version, this.qualifiers, this.subpath);
    }

    public static class StandardTypes {
        public static final String BITBUCKET = "bitbucket";
        public static final String CARGO = "cargo";
        public static final String COMPOSER = "composer";
        public static final String DEBIAN = "deb";
        public static final String DOCKER = "docker";
        public static final String GEM = "gem";
        public static final String GENERIC = "generic";
        public static final String GITHUB = "github";
        public static final String GOLANG = "golang";
        public static final String HEX = "hex";
        public static final String MAVEN = "maven";
        public static final String NPM = "npm";
        public static final String NUGET = "nuget";
        public static final String PYPI = "pypi";
        public static final String RPM = "rpm";
    }
}

