/*
 * Decompiled with CFR 0.152.
 */
package com.sonatype.nexus.git.utils.api;

import com.sonatype.insight.scan.file.FileScanner;
import com.sonatype.insight.scan.file.ManifestContentProcessor;
import com.sonatype.insight.scan.model.ScanConfiguration;
import com.sonatype.nexus.git.utils.api.GitApi;
import com.sonatype.nexus.git.utils.api.GitException;
import com.sonatype.nexus.git.utils.api.NativeGitCommands;
import com.sonatype.nexus.git.utils.api.NativeGitUtils;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.lang.reflect.Field;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import zz.org.apache.commons.lang3.StringUtils;
import zz.org.apache.commons.lang3.Validate;
import zz.org.zeroturnaround.exec.InvalidExitValueException;
import zz.org.zeroturnaround.exec.MessageLogger;
import zz.org.zeroturnaround.exec.MessageLoggers;
import zz.org.zeroturnaround.exec.ProcessExecutor;
import zz.org.zeroturnaround.exec.ProcessResult;

public class NativeGitApi
implements GitApi {
    public static final String GIT_IN_PATH = "git";
    public static final String SSH_PREFIX = "ssh://";
    public static final String HTTPS_PREFIX = "https://";
    public static final String HTTP_PREFIX = "http://";
    public static final String SSH_REGEX = "^(?!http).*@?.+:.*";
    private static final List<String> SUPPORTED_GIT_NAMES = Arrays.asList("git", "git.exe");
    private static final Logger log = LoggerFactory.getLogger(NativeGitApi.class);
    static final Integer EXIT_CODE_SUCCESS = 0;
    private static final Integer EXIT_CODE_FAILURE_128 = 128;
    private static final Set<String> SPARSE_CHECKOUT_FILE_PATTERNS = NativeGitApi.loadSparseCheckoutFilePatterns();
    public static final String URL_SEPARATOR = "/";
    public static final String GIT_SUFFIX = ".git";
    public static final String EMPTY_STRING = "";
    private File tempDir;
    private static final List<String> WINDOWS_TOKEN_CREDENTIAL_CONTENTS = Arrays.asList("@echo \"%GIT_TOKEN%\"");
    private static final List<String> WINDOWS_USERNAME_TOKEN_CREDENTIAL_CONTENTS = Arrays.asList("@set arg=%~1", "@if (%arg:~0,8%)==(Password) echo %GIT_TOKEN%", "@if (%arg:~0,8%)==(Username) echo %GIT_USERNAME%");
    private static final List<String> NIX_TOKEN_CREDENTIAL_CONTENTS = Arrays.asList("#!/bin/sh", "echo \"$GIT_TOKEN\"");
    private static final List<String> NIX_USERNAME_TOKEN_CREDENTIAL_CONTENTS = Arrays.asList("#!/bin/sh", "case \"$1\" in", "Username*) echo \"$GIT_USERNAME\" ;;", "Password*) echo \"$GIT_TOKEN\" ;;", "esac");
    private final int timeout;
    private final String repositoryUrl;
    private final char[] token;
    private final String username;
    private final String gitExecutable;
    private final MessageLogger nativeGitLogger = new NativeGitMessageLogger();

    private static Set<String> loadSparseCheckoutFilePatterns() {
        LinkedHashSet<String> tempFileList = new LinkedHashSet<String>(ManifestContentProcessor.SupportedManifest.getListing());
        tempFileList.add("*.csproj");
        tempFileList.add("*.tfplan");
        for (String archiveType : new FileScanner(null, null).getSupportedArchiveTypes(new ScanConfiguration())) {
            tempFileList.add("*." + archiveType);
        }
        tempFileList.add("*.dll");
        tempFileList.add("*.gem");
        tempFileList.add("*.js");
        tempFileList.add("*.rpm");
        tempFileList.add("build.gradle");
        tempFileList.add("gradle.properties");
        tempFileList.add("go.mod");
        return Collections.unmodifiableSet(tempFileList);
    }

    public NativeGitApi(int timeout, String repositoryUrl, String token, String username, String gitExecutable) {
        this.timeout = timeout;
        this.repositoryUrl = Validate.notEmpty(repositoryUrl, "Repository url is required", new Object[0]);
        this.token = Validate.notEmpty(token, "Token is required", new Object[0]).toCharArray();
        this.username = username;
        this.gitExecutable = gitExecutable != null ? this.validateGitExecutable(gitExecutable) : GIT_IN_PATH;
        log.debug("Created NativeGitApi for repository '{}' using git executable '{}'.", (Object)repositoryUrl, (Object)this.gitExecutable);
    }

    public NativeGitApi(String repositoryUrl, String token, String username, String gitExecutable) {
        this(300, repositoryUrl, token, username, gitExecutable);
    }

    public NativeGitApi(String repositoryUrl, String token, String username) {
        this(repositoryUrl, token, username, null);
    }

    @Override
    public String cloneOrPullRepository(File target, String branch) throws GitException {
        this.validateTarget(target);
        Validate.notEmpty(branch, "Branch is required", new Object[0]);
        if (this.repositoryExists(target)) {
            return this.cleanResetAndPull(target, branch);
        }
        return this.cloneRepo(target, branch);
    }

    @Override
    public String cloneOrPullRepository(File target, String branch, String commitHash) throws GitException {
        this.validateTarget(target);
        Validate.notEmpty(branch, "Branch is required", new Object[0]);
        this.enforceCommitHash(commitHash, "Invalid commit hash provided");
        if (this.repositoryExists(target)) {
            return this.cleanResetAndPull(target, branch, commitHash);
        }
        return this.cloneRepo(target, branch, commitHash);
    }

    @Override
    public String branch(File target, String branch) throws GitException {
        this.checkRepositoryAndRemoteUrl(target);
        if (this.branchExists(target, branch)) {
            throw new GitException("Branch already exists: " + branch);
        }
        return this.gitCreateBranch(target, branch);
    }

    @Override
    public String commit(File target, String username, String email, String message) throws GitException {
        if (StringUtils.isAnyBlank(username, message)) {
            throw new IllegalArgumentException("Parameters username and message are required.");
        }
        this.checkRepositoryAndRemoteUrl(target);
        if (this.isCleanRepository(target)) {
            throw new GitException("Nothing to commit, working tree clean.");
        }
        return this.gitCommit(target, username, email, message);
    }

    @Override
    public String push(File target) throws GitException {
        this.checkRepositoryAndRemoteUrl(target);
        return this.gitPush(target);
    }

    @Override
    public Map<String, String> getHeadCommitsForAllBranches(String repositoryUrl) throws GitException {
        HashMap<String, String> result = new HashMap<String, String>();
        String output = this.gitLsRemote(repositoryUrl);
        if (StringUtils.isNotBlank(output)) {
            this.processLsRemoteOutput(output, result);
        }
        return result;
    }

    @Override
    public String getCommonAncestorCommit(File target, String branchOne, String branchTwo) throws GitException {
        if (StringUtils.isAnyBlank(branchOne, branchTwo)) {
            throw new IllegalArgumentException("Parameters branchOne and branchTwo are required.");
        }
        this.checkRepositoryAndRemoteUrl(target);
        this.gitFetch(target, branchOne, false);
        this.gitFetch(target, branchTwo, false);
        String mergeBase = this.gitMergeBase(target, branchOne, branchTwo);
        return StringUtils.isBlank(mergeBase) ? null : mergeBase;
    }

    private String gitMergeBase(File target, String branchOne, String branchTwo) throws GitException {
        ProcessResult processResult = this.runCommand(target, NativeGitCommands.MERGE_BASE.with("origin/" + branchOne, "origin/" + branchTwo));
        return processResult.outputUTF8().trim();
    }

    private String gitLsRemote(String repositoryUrl) throws GitException {
        File target = this.getTemporaryTarget();
        ProcessResult processResult = this.runCommand(target, NativeGitCommands.LS_REMOTE.with(repositoryUrl));
        return processResult.outputUTF8().trim();
    }

    private File getTemporaryTarget() {
        return this.tempDir != null ? this.tempDir : new File(System.getProperty("java.io.tmpdir"));
    }

    private void processLsRemoteOutput(String output, Map<String, String> result) {
        String[] lines;
        String[] stringArray = lines = output.split("\\r?\\n");
        int n = lines.length;
        int n2 = 0;
        while (n2 < n) {
            String[] parts;
            String line = stringArray[n2];
            if (!line.startsWith("warning: redirecting") && (parts = line.split("\\s+")).length == 2) {
                result.put(parts[1].substring(11), parts[0]);
            }
            ++n2;
        }
    }

    private void checkRepositoryAndRemoteUrl(File target) throws GitException {
        this.validateTarget(target);
        if (!this.repositoryExists(target)) {
            throw new GitException("No repository present in target directory: " + target.getAbsolutePath());
        }
        this.checkAndUpdateRemoteUrl(target);
    }

    private boolean repositoryExists(File target) {
        File nestedGitDir = new File(target, GIT_SUFFIX);
        return nestedGitDir.exists() && nestedGitDir.isDirectory();
    }

    private boolean branchExists(File target, String branch) throws GitException {
        ProcessResult pullResult = this.runCommand(target, NativeGitCommands.HEAD_EXISTS.with("refs/heads/" + branch), Arrays.asList(EXIT_CODE_SUCCESS, EXIT_CODE_FAILURE_128));
        return pullResult.getExitValue() == EXIT_CODE_SUCCESS.intValue();
    }

    private boolean isCleanRepository(File target) throws GitException {
        ProcessResult pullResult = this.runCommand(target, NativeGitCommands.GIT_STATUS.getCommands());
        return EMPTY_STRING.equals(pullResult.outputString());
    }

    private String cleanResetAndPull(File target, String branch) throws GitException {
        return this.cleanResetAndPull(target, branch, null);
    }

    private String cleanResetAndPull(File target, String branch, String commitHash) throws GitException {
        this.checkAndUpdateRemoteUrl(target);
        this.validateRepository(target);
        this.enableSparseCheckout(target);
        this.gitClean(target);
        this.ensureAllBranchesAreFetched(target);
        this.gitFetch(target, branch, commitHash == null);
        this.gitCheckout(target, commitHash == null ? branch : commitHash);
        this.gitReset(target, branch, commitHash);
        return commitHash == null ? this.getHeadRef(target) : commitHash;
    }

    private String cloneRepo(File target, String branch) throws GitException {
        return this.cloneRepo(target, branch, null);
    }

    private String cloneRepo(File target, String branch, String commitHash) throws GitException {
        try {
            if (!this.isDirEmpty(target.toPath())) {
                throw new IllegalStateException("Target directory for new clone is not empty: " + target.getAbsolutePath());
            }
        }
        catch (IOException e) {
            throw new GitException("Failed to check if target path '" + target.getAbsolutePath() + "' is empty", e);
        }
        this.cloneNoCheckout(target, branch, commitHash == null);
        this.enableSparseCheckout(target);
        this.gitCheckout(target, commitHash == null ? branch : commitHash);
        return commitHash == null ? this.getHeadRef(target) : commitHash;
    }

    private void validateRepository(File target) throws GitException {
        ProcessResult pullResult = this.runCommand(target, NativeGitCommands.GET_ORIGIN.getCommands());
        String remoteUrl = StringUtils.trim(pullResult.outputUTF8());
        Validate.isTrue(remoteUrl.equals(this.repositoryUrl), "This repository is configured for a different remote url: %s than the one specified: %s", remoteUrl, this.repositoryUrl);
    }

    private void ensureAllBranchesAreFetched(File target) throws GitException {
        this.runCommand(target, NativeGitCommands.CONFIG_LOCAL.with("remote.origin.fetch", "+refs/heads/*:refs/remotes/origin/*"));
    }

    private void gitClean(File target) throws GitException {
        this.runCommand(target, NativeGitCommands.CLEAN.getCommands());
    }

    private void gitCheckout(File target, String branchOrCommitHash) throws GitException {
        this.runCommand(target, NativeGitCommands.CHECKOUT.with(branchOrCommitHash));
    }

    private void gitFetch(File target, String branch, boolean shallow) throws GitException {
        if (shallow) {
            this.runCommand(target, NativeGitCommands.SHALLOW_FETCH.with(branch));
        } else {
            this.runCommand(target, NativeGitCommands.DEEP_FETCH.with(branch));
        }
    }

    private void gitReset(File target, String branch, String commitHash) throws GitException {
        if (commitHash == null) {
            this.runCommand(target, NativeGitCommands.HARD_RESET.with("origin/" + branch));
        } else {
            this.runCommand(target, NativeGitCommands.HARD_RESET.with(commitHash));
        }
    }

    private String gitCreateBranch(File target, String branch) throws GitException {
        this.runCommand(target, NativeGitCommands.CREATE_BRANCH_AND_CHECKOUT.with(branch));
        return this.getHeadRef(target);
    }

    private String getHeadRef(File target) throws GitException {
        return this.runCommand(target, NativeGitCommands.HEAD_REF.getCommands()).outputUTF8().trim();
    }

    private void cloneNoCheckout(File target, String branch, boolean shallow) throws GitException {
        if (shallow) {
            this.runCommand(target, NativeGitCommands.SHALLOW_CLONE.with(branch, this.repositoryUrl, "."));
        } else {
            this.runCommand(target, NativeGitCommands.DEEP_CLONE.with(branch, this.repositoryUrl, "."));
        }
    }

    private String gitCommit(File target, String username, String email, String message) throws GitException {
        this.configUserEmail(target, email);
        this.runCommand(target, NativeGitCommands.CONFIG_LOCAL.with("user.name", username));
        this.runCommand(target, NativeGitCommands.CONFIG_LOCAL.with("commit.gpgsign", "false"));
        this.runCommand(target, NativeGitCommands.COMMIT.with("-m", message));
        return this.getHeadRef(target);
    }

    ProcessResult configUserEmail(File targetDir, String email) throws GitException {
        if (Objects.equals(EMPTY_STRING, email)) {
            email = this.getOSSpecificDefaultEmail(NativeGitUtils.IS_OS_WINDOWS);
        }
        return this.runCommand(targetDir, NativeGitCommands.CONFIG_LOCAL.with("user.email", email));
    }

    String getOSSpecificDefaultEmail(boolean isWindows) {
        return isWindows ? "^<^>" : "\"<>\"";
    }

    private String gitPush(File target) throws GitException {
        this.runCommand(target, NativeGitCommands.PUSH.getCommands());
        return this.getHeadRef(target);
    }

    private void enableSparseCheckout(File target) throws GitException {
        this.runCommand(target, NativeGitCommands.SPARSE_CHECKOUT.getCommands());
        File dotGit = new File(target, GIT_SUFFIX);
        File info = new File(dotGit, "info");
        File sparse = new File(info, "sparse-checkout");
        try {
            if (!info.exists()) {
                boolean mkdirs = info.mkdirs();
                log.info("Created .git/info directory as it did not exist. Result: {}", (Object)mkdirs);
            }
            Files.write(sparse.toPath(), SPARSE_CHECKOUT_FILE_PATTERNS, StandardCharsets.UTF_8, new OpenOption[0]);
        }
        catch (IOException e) {
            throw new GitException("Unable to write sparse checkout file", e);
        }
    }

    private ProcessResult runCommand(File target, List<String> command) throws GitException {
        return this.runCommand(target, command, Collections.singletonList(EXIT_CODE_SUCCESS));
    }

    private ProcessResult runCommand(File target, List<String> command, List<Integer> exitValues) throws GitException {
        File credentialFile = this.writeCredentialFile();
        HashMap<String, String> environment = new HashMap<String, String>();
        environment.put("GIT_TOKEN", new String(this.token));
        environment.put("GIT_USERNAME", this.username);
        environment.put("GIT_TERMINAL_PROMPT", "0");
        environment.put("GIT_ASKPASS", credentialFile.getAbsolutePath());
        environment.put("SSH_ASKPASS", credentialFile.getAbsolutePath());
        environment.put("GIT_SSH_COMMAND", this.getGitSshCommand());
        List<String> commandWithExecutable = NativeGitUtils.createCommandList(NativeGitUtils.IS_OS_WINDOWS, this.gitExecutable, command);
        String commandMsg = String.join((CharSequence)" ", commandWithExecutable);
        String execId = UUID.randomUUID().toString();
        log.trace("COMMAND ({}): {}", (Object)execId, (Object)commandMsg);
        try {
            ProcessResult processResult = new ProcessExecutor().command(commandWithExecutable).directory(target).environment(environment).timeout(this.timeout, TimeUnit.SECONDS).setMessageLogger(this.nativeGitLogger).exitValues(exitValues.stream().mapToInt(i -> i).toArray()).readOutput(true).execute();
            if (log.isTraceEnabled()) {
                String output = processResult.outputUTF8().trim();
                log.trace("OUTPUT ({}): {}", (Object)execId, (Object)(output.isEmpty() ? "<no output>" : NativeGitApi.redactToken(output)));
            }
            ProcessResult processResult2 = processResult;
            return processResult2;
        }
        catch (InvalidExitValueException e) {
            log.debug("OUTPUT ERROR ({}): {}", (Object)execId, (Object)NativeGitApi.redactToken(e.getResult().outputUTF8().trim()));
            throw this.cleanGitException(String.format("Invalid exit code executing command '%s' in path '%s'.", commandMsg, target.getAbsolutePath()), e);
        }
        catch (Exception e) {
            throw this.cleanGitException(String.format("Failed to run command '%s' in path '%s'.", commandMsg, target.getAbsolutePath()), e);
        }
        finally {
            credentialFile.delete();
        }
    }

    private String getGitSshCommand() {
        return NativeGitUtils.IS_OS_WINDOWS ? "ssh -o BatchMode=yes" : "ssh -oBatchMode=yes";
    }

    private GitException cleanGitException(String message, Exception e) {
        String newMessage = NativeGitApi.redactToken(e.getMessage());
        try {
            Field f = Throwable.class.getDeclaredField("detailMessage");
            f.setAccessible(true);
            f.set(e, newMessage);
        }
        catch (IllegalAccessException | NoSuchFieldException reflectiveOperationException) {
            String error = "An exception occurred but the git token was unable to be redacted so the exception has been suppressed\nThe original exception type was: " + e.getClass().getName() + "\n" + "The original exception message (with token redacted) was: " + newMessage + "\n" + "The original exception stacktrace was: ";
            RuntimeException rte = new RuntimeException(error);
            rte.setStackTrace(e.getStackTrace());
            return new GitException(message, rte);
        }
        return new GitException(message, e);
    }

    private File writeCredentialFile() throws GitException {
        try {
            File tempFile = File.createTempFile("nexus-iq-git-helper", NativeGitUtils.IS_OS_WINDOWS ? ".bat" : ".sh", this.tempDir);
            List<String> askpassContents = StringUtils.isEmpty(this.username) ? (NativeGitUtils.IS_OS_WINDOWS ? WINDOWS_TOKEN_CREDENTIAL_CONTENTS : NIX_TOKEN_CREDENTIAL_CONTENTS) : (NativeGitUtils.IS_OS_WINDOWS ? WINDOWS_USERNAME_TOKEN_CREDENTIAL_CONTENTS : NIX_USERNAME_TOKEN_CREDENTIAL_CONTENTS);
            Files.write(tempFile.toPath(), askpassContents, StandardCharsets.UTF_8, new OpenOption[0]);
            tempFile.setExecutable(true);
            log.trace("Created credentials file: {}", (Object)tempFile.getAbsolutePath());
            return tempFile;
        }
        catch (IOException e) {
            throw new GitException("Unable to write Git Credential helper file", e);
        }
    }

    private String validateGitExecutable(String gitExecutable) {
        boolean valid;
        File git = new File(Objects.requireNonNull(gitExecutable));
        boolean bl = valid = SUPPORTED_GIT_NAMES.contains(git.getName()) && git.exists();
        if (!valid) {
            throw new IllegalArgumentException("gitExecutable must exist and be named 'git' or 'git.exe'. This value is not valid: " + gitExecutable);
        }
        return gitExecutable;
    }

    private static String redactToken(String message) {
        if (message == null) {
            return null;
        }
        return message.replaceAll("(GIT_TOKEN=).*?([,}])", "$1<redacted>$2");
    }

    public void setTempDirectory(File tempDir) {
        if (tempDir != null) {
            if (tempDir.isFile()) {
                throw new RuntimeException(String.valueOf(tempDir.getAbsolutePath()) + " must be a directory.");
            }
            if (!tempDir.exists()) {
                try {
                    Files.createDirectories(tempDir.toPath(), new FileAttribute[0]);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
        }
        this.tempDir = tempDir;
    }

    private void checkAndUpdateRemoteUrl(File target) throws GitException {
        ProcessResult commandResult = this.runCommand(target, NativeGitCommands.GET_ORIGIN.getCommands());
        String remoteUrl = StringUtils.trim(commandResult.outputUTF8());
        if (!this.isRepoNameEqualsToConfiguredRepoName(remoteUrl)) {
            return;
        }
        if (this.isHttpCloneUrl(remoteUrl) && this.isSshCloneUrl(this.repositoryUrl) || this.isSshCloneUrl(remoteUrl) && this.isHttpCloneUrl(this.repositoryUrl)) {
            this.runCommand(target, NativeGitCommands.SET_ORIGIN.with(this.repositoryUrl));
        }
    }

    private boolean isRepoNameEqualsToConfiguredRepoName(String remoteUrl) {
        String remoteRepoName = this.getRepoNameFromUrl(remoteUrl);
        String configuredRepoName = this.getRepoNameFromUrl(this.repositoryUrl);
        return configuredRepoName.equals(remoteRepoName);
    }

    private String getRepoNameFromUrl(String url) {
        if (StringUtils.isEmpty(url)) {
            return null;
        }
        String[] urlParts = url.split(URL_SEPARATOR);
        return urlParts[urlParts.length - 1].replace(GIT_SUFFIX, EMPTY_STRING);
    }

    private boolean isHttpCloneUrl(String cloneUrl) {
        return cloneUrl.startsWith(HTTPS_PREFIX) || cloneUrl.startsWith(HTTP_PREFIX);
    }

    private boolean isSshCloneUrl(String cloneUrl) {
        return cloneUrl.matches(SSH_REGEX) || cloneUrl.startsWith(SSH_PREFIX);
    }

    static class NativeGitMessageLogger
    implements MessageLogger {
        MessageLogger logger = MessageLoggers.DEBUG;

        NativeGitMessageLogger() {
        }

        @Override
        public void message(Logger log, String format, Object ... arguments) {
            this.logger.message(log, NativeGitApi.redactToken(format), arguments);
        }
    }
}

