/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.repository.docker.internal.orient;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Named;
import org.joda.time.DateTime;
import org.sonatype.nexus.blobstore.api.Blob;
import org.sonatype.nexus.blobstore.api.BlobStoreManager;
import org.sonatype.nexus.common.entity.EntityMetadata;
import org.sonatype.nexus.common.stateguard.Guarded;
import org.sonatype.nexus.logging.task.ProgressLogIntervalHelper;
import org.sonatype.nexus.repository.FacetSupport;
import org.sonatype.nexus.repository.Repository;
import org.sonatype.nexus.repository.docker.DockerGCFacet;
import org.sonatype.nexus.repository.docker.internal.AssetKind;
import org.sonatype.nexus.repository.docker.internal.DockerDigest;
import org.sonatype.nexus.repository.docker.internal.DockerFacetUtils;
import org.sonatype.nexus.repository.docker.internal.MediaType;
import org.sonatype.nexus.repository.docker.internal.V2Exception;
import org.sonatype.nexus.repository.docker.internal.V2Manifest;
import org.sonatype.nexus.repository.docker.internal.V2ManifestUtil;
import org.sonatype.nexus.repository.docker.internal.orient.DockerFacetDatabaseUtils;
import org.sonatype.nexus.repository.manager.RepositoryManager;
import org.sonatype.nexus.repository.storage.Asset;
import org.sonatype.nexus.repository.storage.Bucket;
import org.sonatype.nexus.repository.storage.Component;
import org.sonatype.nexus.repository.storage.MissingBlobException;
import org.sonatype.nexus.repository.storage.Query;
import org.sonatype.nexus.repository.storage.StorageFacet;
import org.sonatype.nexus.repository.storage.StorageTx;
import org.sonatype.nexus.repository.transaction.TransactionalDeleteBlob;
import org.sonatype.nexus.scheduling.CancelableHelper;
import org.sonatype.nexus.transaction.Transactional;
import org.sonatype.nexus.transaction.UnitOfWork;

@Named
public class DockerGCFacetImpl
extends FacetSupport
implements DockerGCFacet {
    private static final int PROGRESS_LOG_INTERVAL = 60;
    private static final String LAYER_ID_ATTRIBUTE = "attributes.docker.layerId";
    private final V2ManifestUtil v2ManifestUtil;
    private final RepositoryManager repositoryManager;
    private final BlobStoreManager blobStoreManager;
    private final long batchSize;
    private final long maxAssets;

    @Inject
    public DockerGCFacetImpl(V2ManifestUtil v2ManifestUtil, RepositoryManager repositoryManager, BlobStoreManager blobStoreManager, @Named(value="${nexus.removeUnusedLayers.batchSize:-500}") long batchSize, @Named(value="${nexus.docker.gc.maxAssets:-50000}") long maxAssets) {
        this.v2ManifestUtil = (V2ManifestUtil)Preconditions.checkNotNull((Object)v2ManifestUtil);
        this.repositoryManager = (RepositoryManager)Preconditions.checkNotNull((Object)repositoryManager);
        this.blobStoreManager = (BlobStoreManager)Preconditions.checkNotNull((Object)blobStoreManager);
        this.batchSize = batchSize;
        this.maxAssets = maxAssets;
    }

    @Override
    @Guarded(by={"STARTED"})
    public void deleteUnusedManifestsAndImages(int deployOffsetHours) {
        Repository repository = this.getRepository();
        this.log.info("Garbage collection starting on repository: {}", (Object)repository);
        boolean isStorageAvailable = false;
        try {
            String blobStoreName = (String)((Map)Objects.requireNonNull(repository.getConfiguration().getAttributes()).get("storage")).get("blobStoreName");
            isStorageAvailable = Objects.requireNonNull(this.blobStoreManager.get(blobStoreName)).isStorageAvailable();
        }
        catch (NullPointerException nullPointerException) {}
        if (!isStorageAvailable) {
            this.log.error("Garbage collection skipped for repository: {} (its Blob Store is unavailable)", (Object)repository);
            return;
        }
        UnitOfWork.beginBatch((Supplier)((StorageFacet)this.facet(StorageFacet.class)).txSupplier());
        try {
            try {
                this.processRepository(repository, deployOffsetHours);
                this.log.info("Garbage collection completed on repository: {}", (Object)repository);
            }
            catch (IOException e) {
                this.log.error("Garbage collection failed on repository: {}", (Object)repository, (Object)e);
                throw new UncheckedIOException(e);
            }
        }
        finally {
            UnitOfWork.end();
        }
    }

    @TransactionalDeleteBlob
    protected void processRepository(Repository repository, int deployOffsetHours) throws IOException {
        StorageTx tx = (StorageTx)UnitOfWork.currentTx();
        long offsetTime = DateTime.now().minusHours(deployOffsetHours).getMillis();
        this.handleV1Assets(tx, repository, offsetTime);
        this.handleV2Assets(tx, repository, offsetTime);
    }

    @VisibleForTesting
    void handleV1Assets(StorageTx tx, Repository repository, long offsetTime) {
        CancelableHelper.checkCancellation();
        ProgressLogIntervalHelper progressLogger = new ProgressLogIntervalHelper(this.log, 60);
        this.log.info("Running GC for v1 assets on {}", (Object)repository.getName());
        Iterable components = tx.findComponents(Query.builder().where(LAYER_ID_ATTRIBUTE).isNotNull().build(), Collections.singleton(repository));
        HashSet usedLayerIds = new HashSet();
        for (Component component : components) {
            String layerId;
            CancelableHelper.checkCancellation();
            if (!component.lastUpdated().isBefore(offsetTime)) continue;
            List layerAncestry = (List)component.formatAttributes().get("layerAncestry");
            if (layerAncestry == null && (layerId = (String)component.formatAttributes().get("layerId")) != null) {
                Asset metadataAsset = DockerFacetDatabaseUtils.findAsset(tx, tx.findBucket(this.getRepository()), DockerFacetUtils.v1layerMetadataName(layerId));
                layerAncestry = (List)metadataAsset.formatAttributes().get("layerAncestry");
            }
            if (layerAncestry == null) continue;
            usedLayerIds.addAll(layerAncestry);
        }
        CancelableHelper.checkCancellation();
        Iterable assets = tx.browseAllPartiallyByLimit(tx.findBucket(repository));
        long count = 0L;
        long deleted = 0L;
        for (Asset asset : assets) {
            CancelableHelper.checkCancellation();
            if (this.isV1Asset(asset) && asset.lastUpdated().isBefore(offsetTime) && !usedLayerIds.contains(asset.formatAttributes().get("layerId").toString())) {
                this.log.debug("Found layer to delete {}", (Object)asset);
                EntityMetadata metadata = asset.getEntityMetadata();
                tx.deleteAsset(asset);
                this.maybeCommit(++deleted, tx);
                asset.setEntityMetadata(metadata);
            }
            progressLogger.info("Deleting unused V1 assets on {}. checked {}, deleted {}", new Object[]{repository.getName(), ++count, deleted});
        }
        this.log.info("Finished GC for v1 assets on {}. checked {}, deleted {}", new Object[]{repository.getName(), count, deleted});
    }

    @VisibleForTesting
    void handleV2Assets(StorageTx tx, Repository repository, long offsetTime) {
        CancelableHelper.checkCancellation();
        this.log.info("Running GC for v2 assets on {}", (Object)repository.getName());
        ProgressLogIntervalHelper progressLogger = new ProgressLogIntervalHelper(this.log, 60);
        long deleted = 0L;
        HashSet<String> taggedManifestDigests = new HashSet<String>();
        HashSet<Asset> digestManifests = new HashSet<Asset>();
        Set<String> usedLayerDigests = this.fetchDataForRepository(repository, taggedManifestDigests, digestManifests);
        deleted = this.maybeDeleteDigestManifests(tx, offsetTime, deleted, taggedManifestDigests, digestManifests);
        Iterable assets = tx.browseAllPartiallyByLimit(tx.findBucket(repository));
        List<Asset> unusedAssets = this.findUnusedAssets(offsetTime, usedLayerDigests, assets, repository);
        this.checkForReferencesInOtherRepositories(tx, repository, unusedAssets);
        this.log.info("Found {} v2 assets to delete on {}", (Object)unusedAssets.size(), (Object)repository.getName());
        for (Asset asset : unusedAssets) {
            CancelableHelper.checkCancellation();
            tx.deleteAsset(asset);
            this.maybeCommit(++deleted, tx);
            progressLogger.info("Deleting unused V2 assets on {}. deleted {}", new Object[]{repository.getName(), deleted});
        }
        this.log.info("Finished GC for v2 assets on {}. deleted {}", (Object)repository.getName(), (Object)deleted);
    }

    private void checkForReferencesInOtherRepositories(StorageTx tx, Repository repository, List<Asset> unusedAssets) {
        for (Repository repo : this.repositoryManager.browse()) {
            if (!repo.getFormat().getValue().equals("docker") || repo.getName().equals(repository.getName()) || repo.getType().getValue().equals("group")) continue;
            this.log.debug("Checking {} for references to {}", (Object)repo.getName(), (Object)repository.getName());
            Set<String> usedLayerDigests = this.getExternalUsedLayersDigests(tx, repo);
            ListIterator<Asset> it = unusedAssets.listIterator();
            while (it.hasNext()) {
                CancelableHelper.checkCancellation();
                Asset asset = it.next();
                if (!usedLayerDigests.contains(asset.formatAttributes().get("content_digest").toString())) continue;
                this.log.debug("Found asset referenced in another repository {}", (Object)asset);
                it.remove();
            }
        }
    }

    private Set<String> getExternalUsedLayersDigests(StorageTx tx, Repository repository) {
        UnitOfWork currentUnitOfWork = UnitOfWork.pause();
        try {
            Set<String> set;
            tx.commit();
            UnitOfWork.begin((Supplier)((StorageFacet)repository.facet(StorageFacet.class)).txSupplier());
            try {
                set = this.fetchDataForRepository(repository, null, null);
            }
            catch (Throwable throwable) {
                UnitOfWork.end();
                throw throwable;
            }
            UnitOfWork.end();
            return set;
        }
        finally {
            UnitOfWork.resume((UnitOfWork)currentUnitOfWork);
            tx.begin();
        }
    }

    private List<Asset> findUnusedAssets(long offsetTime, Set<String> usedLayerDigests, Iterable<Asset> assets, Repository repository) {
        ProgressLogIntervalHelper progressLogger = new ProgressLogIntervalHelper(this.log, 60);
        ArrayList<Asset> unusedAssets = new ArrayList<Asset>();
        long count = 0L;
        for (Asset asset : assets) {
            CancelableHelper.checkCancellation();
            if (this.isBlob(asset) && asset.lastUpdated().isBefore(offsetTime) && !usedLayerDigests.contains(asset.formatAttributes().get("content_digest").toString())) {
                this.log.debug("Found possible asset to delete {}", (Object)asset);
                unusedAssets.add(asset);
                if ((long)unusedAssets.size() >= this.maxAssets) break;
            }
            progressLogger.info("Finding v2 assets to delete on {}. checked {}, found {}", new Object[]{repository.getName(), ++count, unusedAssets.size()});
        }
        return unusedAssets;
    }

    private long maybeDeleteDigestManifests(StorageTx tx, long offsetTime, long deleted, Set<String> taggedManifestDigests, Set<Asset> digestManifests) {
        for (Asset asset : digestManifests) {
            CancelableHelper.checkCancellation();
            if (!asset.lastUpdated().isBefore(offsetTime) || taggedManifestDigests.contains(asset.formatAttributes().get("content_digest").toString())) continue;
            this.log.debug("Found manifest to delete {}", (Object)asset);
            tx.deleteAsset(asset);
            this.maybeCommit(++deleted, tx);
        }
        return deleted;
    }

    @Transactional
    protected Set<String> fetchDataForRepository(Repository repository, @Nullable Set<String> taggedManifestDigests, @Nullable Set<Asset> digestManifests) {
        ProgressLogIntervalHelper progressLogger = new ProgressLogIntervalHelper(this.log, 60);
        HashSet<String> usedLayerDigests = new HashSet<String>();
        StorageTx tx = (StorageTx)UnitOfWork.currentTx();
        Bucket bucket = tx.findBucket(repository);
        Iterable assets = tx.browseAllPartiallyByLimit(bucket);
        long count = 0L;
        for (Asset asset : assets) {
            CancelableHelper.checkCancellation();
            if (this.isManifest(asset)) {
                if (this.isDigestManifest(asset.name())) {
                    if (digestManifests != null) {
                        digestManifests.add(asset);
                    }
                } else {
                    if (taggedManifestDigests != null) {
                        taggedManifestDigests.add(asset.formatAttributes().get("content_digest").toString());
                    }
                    this.fetchDataForManifest(tx, usedLayerDigests, taggedManifestDigests, bucket, asset);
                }
            }
            progressLogger.info("Fetching data for {}. checked {}", new Object[]{repository.getName(), ++count});
        }
        this.log.info("Fetched data for {}. checked {}", (Object)repository, (Object)count);
        return usedLayerDigests;
    }

    private void fetchDataForManifest(StorageTx tx, Set<String> usedLayerDigests, @Nullable Set<String> taggedManifestDigests, Bucket bucket, Asset assetManifest) {
        try {
            V2Manifest manifest = this.v2ManifestUtil.readManifest(() -> ((Blob)tx.requireBlob(assetManifest.blobRef())).getInputStream(), assetManifest.name(), assetManifest.contentType());
            if (MediaType.IMAGE_INDEX.matches(manifest.getMediaType())) {
                this.fetchDataForManifestList(assetManifest, manifest, tx, bucket, usedLayerDigests, taggedManifestDigests);
            }
            for (DockerDigest dockerDigest : manifest.referencedDigests()) {
                usedLayerDigests.add(dockerDigest.toString());
            }
        }
        catch (IOException | V2Exception.ManifestInvalid | MissingBlobException e) {
            this.log.warn("Unable to read V2 Manifest for asset {}, {}", (Object)assetManifest, (Object)e.getMessage());
            this.log.debug("Exception is", e);
        }
    }

    private void fetchDataForManifestList(Asset asset, V2Manifest manifest, StorageTx tx, Bucket bucket, Set<String> usedLayerDigests, @Nullable Set<String> taggedManifestDigests) {
        List<DockerDigest> listedManifests = manifest.getListedManifests();
        for (DockerDigest listedManifest : listedManifests) {
            String listedManifestName = this.getListedManifestName(asset, listedManifest);
            Asset manifestAsset = DockerFacetDatabaseUtils.findAsset(tx, bucket, listedManifestName);
            if (manifestAsset == null) {
                this.log.debug("Cannot find manifest with name: {}", (Object)listedManifestName);
                continue;
            }
            try {
                V2Manifest v2Manifest = this.v2ManifestUtil.readManifest(() -> ((Blob)tx.requireBlob(manifestAsset.blobRef())).getInputStream(), asset.name(), asset.contentType());
                usedLayerDigests.addAll(v2Manifest.referencedDigests().stream().map(DockerDigest::toString).collect(Collectors.toList()));
            }
            catch (IOException | V2Exception.ManifestInvalid e) {
                this.log.error("Failed to deserialize manifest with name: {}, skipping...", (Object)listedManifestName, (Object)e);
            }
        }
        if (taggedManifestDigests != null) {
            taggedManifestDigests.addAll(listedManifests.stream().map(DockerDigest::toString).collect(Collectors.toList()));
        }
    }

    private boolean isDigestManifest(String name) {
        return name.contains("manifests/sha256:");
    }

    private void maybeCommit(long count, StorageTx tx) {
        if (count % this.batchSize == 0L) {
            this.log.debug("Committing batch delete");
            tx.commit();
            tx.begin();
        }
    }

    private String getListedManifestName(Asset manifestListAsset, DockerDigest listedManifest) {
        return String.valueOf(manifestListAsset.name().substring(0, manifestListAsset.name().lastIndexOf("/") + 1)) + listedManifest.toString();
    }

    private boolean isV1Asset(Asset asset) {
        String assetKind = (String)asset.formatAttributes().get("asset_kind", String.class);
        return AssetKind.LAYER_CONTENT.name().equals(assetKind) || AssetKind.LAYER_METADATA.name().equals(assetKind);
    }

    private boolean isBlob(Asset asset) {
        String assetKind = (String)asset.formatAttributes().get("asset_kind", String.class);
        return AssetKind.BLOB.name().equals(assetKind);
    }

    private boolean isManifest(Asset asset) {
        String assetKind = (String)asset.formatAttributes().get("asset_kind", String.class);
        return AssetKind.MANIFEST.name().equals(assetKind);
    }
}

