/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.repository.http;

import com.google.common.base.Preconditions;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.http.client.utils.DateUtils;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonatype.nexus.repository.view.Context;
import org.sonatype.nexus.repository.view.Headers;
import org.sonatype.nexus.repository.view.Request;
import org.sonatype.nexus.repository.view.Response;
import org.sonatype.nexus.repository.view.Status;

public class HttpConditions {
    private static final Logger log = LoggerFactory.getLogger(HttpConditions.class);
    private static final String HTTP_CONDITIONS = String.valueOf(HttpConditions.class.getName()) + ".conditions";
    private static final List<String> SUPPORTED_HEADERS = Arrays.asList("If-Modified-Since", "If-Unmodified-Since", "If-Match", "If-None-Match");

    private HttpConditions() {
    }

    public static boolean isConditional(Request request) {
        Set headerNames = request.getHeaders().names();
        return headerNames.contains("If-Modified-Since") || headerNames.contains("If-Unmodified-Since") || headerNames.contains("If-Match") || headerNames.contains("If-None-Match");
    }

    @Nonnull
    public static Request makeUnconditional(@Nonnull Request request) {
        Preconditions.checkNotNull((Object)request);
        Headers stashedHeaders = new Headers();
        for (String httpHeader : SUPPORTED_HEADERS) {
            List headerValues = request.getHeaders().getAll(httpHeader);
            if (headerValues != null) {
                stashedHeaders.set(httpHeader, headerValues);
            }
            request.getHeaders().remove(httpHeader);
        }
        request.getAttributes().set(HTTP_CONDITIONS, (Object)stashedHeaders);
        return request;
    }

    @Nonnull
    public static Request makeConditional(@Nonnull Request request) {
        Preconditions.checkNotNull((Object)request);
        Headers stashedHeaders = HttpConditions.getConditionalHeaders(request);
        for (Map.Entry entry : stashedHeaders.entries()) {
            request.getHeaders().set((String)entry.getKey(), stashedHeaders.getAll((String)entry.getKey()));
        }
        return request;
    }

    private static Headers getConditionalHeaders(Request request) {
        return (Headers)((Object)request.getAttributes().require(HTTP_CONDITIONS, Headers.class));
    }

    public static Optional<Response> maybeCreateConditionalResponse(Context context, Response getResponse) {
        boolean modifiedAfterSpecifiedDate;
        Headers headers = HttpConditions.getConditionalHeaders(context.getRequest());
        Optional<String> ifMatchHeader = Optional.ofNullable(headers.get("If-Match"));
        Optional<String> ifUnmodifiedHeader = Optional.ofNullable(headers.get("If-Unmodified-Since"));
        Optional<String> ifNoneMatchHeader = Optional.ofNullable(headers.get("If-None-Match"));
        Optional<String> ifModifiedHeader = Optional.ofNullable(headers.get("If-Modified-Since"));
        Optional<DateTime> lastModified = HttpConditions.parseDateHeader(getResponse.getHeaders().get("Last-Modified"));
        if (ifMatchHeader.isPresent()) {
            if (!HttpConditions.doesEtagMatch(ifMatchHeader.get(), getResponse)) {
                log.debug("ETag: {} does not If-Unmodified: {}", (Object)getResponse.getHeaders().get("ETag"), (Object)ifMatchHeader.get());
                return Optional.of(HttpConditions.buildPreconditionFailed(getResponse));
            }
        } else if (ifUnmodifiedHeader.isPresent() && lastModified.isPresent()) {
            modifiedAfterSpecifiedDate = ifUnmodifiedHeader.flatMap(HttpConditions::parseDateHeader).map(arg_0 -> ((DateTime)lastModified.get()).isAfter(arg_0)).orElse(false);
            if (modifiedAfterSpecifiedDate) {
                log.debug("Precondition Failed - LastModified {} is after If-Unmodified-Since {}", (Object)lastModified.get(), (Object)ifUnmodifiedHeader.get());
                return Optional.of(HttpConditions.buildPreconditionFailed(getResponse));
            }
        }
        if (ifNoneMatchHeader.isPresent()) {
            if (HttpConditions.doesEtagMatch(ifNoneMatchHeader.get(), getResponse)) {
                log.debug("ETag: {} does not If-None-Match: {}", (Object)getResponse.getHeaders().get("ETag"), (Object)ifNoneMatchHeader.get());
                return Optional.of(HttpConditions.getOrHead(context.getRequest()) ? HttpConditions.buildNotModified(getResponse) : HttpConditions.buildPreconditionFailed(getResponse));
            }
        } else if (ifModifiedHeader.isPresent() && lastModified.isPresent() && HttpConditions.getOrHead(context.getRequest())) {
            modifiedAfterSpecifiedDate = ifModifiedHeader.flatMap(HttpConditions::parseDateHeader).map(arg_0 -> ((DateTime)lastModified.get()).isAfter(arg_0)).orElse(true);
            if (!modifiedAfterSpecifiedDate) {
                log.debug("Not Modified - LastModified {} is after If-Modified-Since {}", (Object)lastModified.get(), (Object)ifModifiedHeader.get());
                return Optional.of(HttpConditions.buildNotModified(getResponse));
            }
        }
        return Optional.empty();
    }

    private static boolean getOrHead(Request request) {
        String action = request.getAction();
        return "GET".equals(action) || "HEAD".equals(action);
    }

    private static Response buildPreconditionFailed(Response response) {
        return new Response.Builder().copy(response).status(Status.failure(412)).payload(null).build();
    }

    private static Response buildNotModified(Response response) {
        Response.Builder responseBuilder = new Response.Builder().status(Status.success(304));
        Optional.ofNullable(response.getHeaders().get("ETag")).ifPresent(eTag -> {
            Response.Builder builder2 = responseBuilder.header("ETag", (String)eTag);
        });
        return responseBuilder.build();
    }

    private static boolean doesEtagMatch(String matchHeader, Response response) {
        if ("*".equals(matchHeader)) {
            return true;
        }
        return Optional.ofNullable(response.getHeaders().get("ETag")).map(matchHeader::contains).orElse(false);
    }

    private static Optional<DateTime> parseDateHeader(@Nullable String httpDate) {
        return Optional.ofNullable(httpDate).map(DateUtils::parseDate).map(Date::getTime).map(DateTime::new);
    }
}

