/*
 * Decompiled with CFR 0.152.
 */
package org.sonatype.nexus.blobstore.restore.datastore;

import com.google.common.base.Preconditions;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import javax.inject.Inject;
import javax.inject.Named;
import org.sonatype.nexus.blobstore.api.Blob;
import org.sonatype.nexus.blobstore.api.BlobAttributes;
import org.sonatype.nexus.blobstore.api.BlobId;
import org.sonatype.nexus.blobstore.api.BlobStore;
import org.sonatype.nexus.blobstore.api.BlobStoreManager;
import org.sonatype.nexus.blobstore.api.BlobStoreUsageChecker;
import org.sonatype.nexus.blobstore.restore.RestoreBlobStrategy;
import org.sonatype.nexus.blobstore.restore.datastore.IntegrityCheckStrategy;
import org.sonatype.nexus.common.log.DryRunPrefix;
import org.sonatype.nexus.logging.task.ProgressLogIntervalHelper;
import org.sonatype.nexus.repository.Repository;
import org.sonatype.nexus.repository.content.Asset;
import org.sonatype.nexus.repository.content.maintenance.MaintenanceService;
import org.sonatype.nexus.repository.manager.RepositoryManager;
import org.sonatype.nexus.repository.types.GroupType;
import org.sonatype.nexus.scheduling.Cancelable;
import org.sonatype.nexus.scheduling.TaskSupport;

@Named
public class RestoreMetadataTask
extends TaskSupport
implements Cancelable {
    private final BlobStoreManager blobStoreManager;
    private final RepositoryManager repositoryManager;
    private final Map<String, RestoreBlobStrategy> restoreBlobStrategies;
    private final Map<String, IntegrityCheckStrategy> integrityCheckStrategies;
    private final BlobStoreUsageChecker blobStoreUsageChecker;
    private final DryRunPrefix dryRunPrefix;
    private final IntegrityCheckStrategy defaultIntegrityCheckStrategy;
    private final MaintenanceService maintenanceService;

    @Inject
    public RestoreMetadataTask(BlobStoreManager blobStoreManager, RepositoryManager repositoryManager, Map<String, RestoreBlobStrategy> restoreBlobStrategies, BlobStoreUsageChecker blobStoreUsageChecker, DryRunPrefix dryRunPrefix, Map<String, IntegrityCheckStrategy> integrityCheckStrategies, MaintenanceService maintenanceService) {
        this.blobStoreManager = (BlobStoreManager)Preconditions.checkNotNull((Object)blobStoreManager);
        this.repositoryManager = (RepositoryManager)Preconditions.checkNotNull((Object)repositoryManager);
        this.restoreBlobStrategies = (Map)Preconditions.checkNotNull(restoreBlobStrategies);
        this.blobStoreUsageChecker = (BlobStoreUsageChecker)Preconditions.checkNotNull((Object)blobStoreUsageChecker);
        this.dryRunPrefix = (DryRunPrefix)Preconditions.checkNotNull((Object)dryRunPrefix);
        this.defaultIntegrityCheckStrategy = (IntegrityCheckStrategy)Preconditions.checkNotNull((Object)integrityCheckStrategies.get("default"));
        this.integrityCheckStrategies = (Map)Preconditions.checkNotNull(integrityCheckStrategies);
        this.maintenanceService = (MaintenanceService)Preconditions.checkNotNull((Object)maintenanceService);
    }

    public String getMessage() {
        return "Uses blobs in a blobstore to restore assets to a repository";
    }

    protected Void execute() throws Exception {
        String blobStoreId = (String)Preconditions.checkNotNull((Object)this.getConfiguration().getString("blobstoreName"));
        boolean dryRun = this.getConfiguration().getBoolean("dryRun", false);
        boolean restoreBlobs = this.getConfiguration().getBoolean("restoreBlobs", false);
        boolean undeleteBlobs = this.getConfiguration().getBoolean("undeleteBlobs", false);
        boolean integrityCheck = this.getConfiguration().getBoolean("integrityCheck", false);
        Integer sinceDays = this.getConfiguration().getInteger("sinceDays", -1);
        BlobStore blobStore = this.blobStoreManager.get(blobStoreId);
        this.restore(blobStore, restoreBlobs, undeleteBlobs, dryRun, sinceDays);
        this.blobStoreIntegrityCheck(integrityCheck, blobStoreId, dryRun);
        return null;
    }

    private void restore(BlobStore blobStore, boolean restore, boolean undelete, boolean dryRun, Integer sinceDays) {
        if (!restore && !undelete) {
            this.log.warn("No repair/restore operations selected");
            return;
        }
        String logPrefix = dryRun ? this.dryRunPrefix.get() : "";
        long processed = 0L;
        long undeleted = 0L;
        boolean updateAssets = !dryRun && restore;
        HashSet<Repository> touchedRepositories = new HashSet<Repository>();
        if (dryRun) {
            this.log.info("{}Actions will be logged, but no changes will be made.", (Object)logPrefix);
        }
        Throwable throwable = null;
        Object var14_13 = null;
        try (ProgressLogIntervalHelper progressLogger = new ProgressLogIntervalHelper(this.log, 60);){
            for (BlobId blobId : this.getBlobIdStream(blobStore, sinceDays)) {
                try {
                    if (this.isCanceled()) {
                        this.log.info("Restore metadata task for {} was canceled", (Object)blobStore.getBlobStoreConfiguration().getName());
                        break;
                    }
                    Optional<Context> optionalContext = this.buildContext(blobStore, blobId);
                    if (optionalContext.isPresent()) {
                        Context context = optionalContext.get();
                        if (restore && context.restoreBlobStrategy != null && !context.blobAttributes.isDeleted()) {
                            context.restoreBlobStrategy.restore(context.properties, context.blob, context.blobStore, dryRun);
                        }
                        if (undelete && blobStore.undelete(this.blobStoreUsageChecker, context.blobId, context.blobAttributes, dryRun)) {
                            ++undeleted;
                        }
                        if (updateAssets) {
                            touchedRepositories.add(context.repository);
                        }
                    }
                    progressLogger.info("{}Elapsed time: {}, processed: {}, un-deleted: {}", new Object[]{logPrefix, progressLogger.getElapsed(), ++processed, undeleted});
                }
                catch (Exception e) {
                    this.log.error("Error restoring blob {}", (Object)blobId, (Object)e);
                }
            }
            this.updateAssets(touchedRepositories, updateAssets);
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    private Iterable<BlobId> getBlobIdStream(BlobStore store, Integer sinceDays) {
        if (Objects.isNull(sinceDays) || sinceDays < 0) {
            this.log.info("Will process all blobs");
            return store.getBlobIdStream()::iterator;
        }
        return store.getBlobIdUpdatedSinceStream(sinceDays.intValue())::iterator;
    }

    private void updateAssets(Set<Repository> repositories, boolean updateAssets) {
        for (Repository repository : repositories) {
            if (this.isCanceled()) break;
            Optional.ofNullable(this.restoreBlobStrategies.get(repository.getFormat().getValue())).ifPresent(strategy -> strategy.after(updateAssets, repository));
        }
    }

    private void blobStoreIntegrityCheck(boolean integrityCheck, String blobStoreId, boolean dryRun) {
        if (!integrityCheck) {
            this.log.warn("Integrity check operation not selected");
            return;
        }
        BlobStore blobStore = this.blobStoreManager.get(blobStoreId);
        if (blobStore == null) {
            this.log.error("Unable to find blob store '{}' in the blob store manager", (Object)blobStoreId);
            return;
        }
        Iterable repositories = this.repositoryManager.browseForBlobStore(blobStoreId);
        List syncList = Collections.synchronizedList(StreamSupport.stream(repositories.spliterator(), false).collect(Collectors.toList()));
        syncList.stream().filter(Objects::nonNull).filter(repository -> !(repository.getType() instanceof GroupType)).filter(Repository::isStarted).forEach(repository -> this.integrityCheckStrategies.getOrDefault(repository.getFormat().getValue(), this.defaultIntegrityCheckStrategy).check((Repository)repository, blobStore, () -> ((RestoreMetadataTask)this).isCanceled(), a -> this.integrityCheckFailedHandler((Repository)repository, (Asset)a, dryRun)));
    }

    protected void integrityCheckFailedHandler(Repository repository, Asset asset, boolean isDryRun) {
        this.log.info("{}Removing asset {} from repository {}, blob integrity check failed", new Object[]{isDryRun ? this.dryRunPrefix.get() : "", asset.path(), repository.getName()});
        if (!isDryRun) {
            this.maintenanceService.deleteAsset(repository, asset);
        }
    }

    private Optional<Context> buildContext(BlobStore blobStore, BlobId blobId) {
        return Optional.of(new Context(blobStore, blobId)).map(c -> c.blob(c.blobStore.get(c.blobId, true))).map(c -> c.blobAttributes(c.blobStore.getBlobAttributes(c.blobId))).map(c -> c.properties(c.blobAttributes.getProperties())).map(c -> c.repositoryName(c.properties.getProperty("@Bucket.repo-name"))).map(c -> c.repository(this.repositoryManager.get(c.repositoryName))).map(c -> c.restoreBlobStrategy(this.restoreBlobStrategies.get(c.repository.getFormat().getValue())));
    }

    private static class Context {
        final BlobStore blobStore;
        final BlobId blobId;
        Blob blob;
        BlobAttributes blobAttributes;
        Properties properties;
        String repositoryName;
        Repository repository;
        RestoreBlobStrategy restoreBlobStrategy;

        Context(BlobStore blobStore, BlobId blobId) {
            this.blobStore = (BlobStore)Preconditions.checkNotNull((Object)blobStore);
            this.blobId = (BlobId)Preconditions.checkNotNull((Object)blobId);
        }

        Context blob(Blob blob) {
            if (blob == null) {
                return null;
            }
            this.blob = blob;
            return this;
        }

        Context blobAttributes(BlobAttributes blobAttributes) {
            if (blobAttributes == null) {
                return null;
            }
            this.blobAttributes = blobAttributes;
            return this;
        }

        Context properties(Properties properties) {
            if (properties == null) {
                return null;
            }
            this.properties = properties;
            return this;
        }

        Context repositoryName(String repositoryName) {
            if (repositoryName == null) {
                return null;
            }
            this.repositoryName = repositoryName;
            return this;
        }

        Context repository(Repository repository) {
            if (repository == null) {
                return null;
            }
            this.repository = repository;
            return this;
        }

        Context restoreBlobStrategy(RestoreBlobStrategy restoreBlobStrategy) {
            this.restoreBlobStrategy = restoreBlobStrategy;
            return this;
        }
    }
}

